local venuscore = require "venuscore"
local apollonode = require "apolloutility.apollonode"
local apolloengine = require "apolloengine"
local fbxfunc = require "fbxtool"
local venusjson = require "venusjson"
local utility = require "emoji.emojiUtility"
local paths = require "paths"
local libmasquerade = require "libmasque"
local web = require "cartoon.web"

local main = {}

local material_path = "comm:documents/material/nl_simple.material" -- just for create the mesh
  
local blendshape_names = {
  {
    "browInnerUp", 
    "browOuterUp_L", 
    "browOuterUp_R", 
    "browDown_L", 
    "browDown_R", 
    "cheekPuff", 
    "eyeBlink_L", 
    "eyeBlink_R", 
    "eyeLookDown_L", 
    "eyeLookDown_R", 
    "eyeLookIn_L", 
    "eyeLookIn_R", 
    "eyeLookOut_L", 
    "eyeLookOut_R", 
    "eyeLookUp_L", 
    "eyeLookUp_R", 
    "eyeSquint_L", 
    "eyeSquint_R", 
    "eyeWide_L", 
    "eyeWide_R", 
    "jawForward", 
    "jawLeft", 
    "jawRight", 
    "jawOpen", 
    "jawOpenMouthClose", 
    "mouthDimple_L", 
    "mouthDimple_R", 
    "mouthLeft", 
    "mouthRight", 
    "mouthFrown_L", 
    "mouthFrown_R", 
    "mouthFunnel", 
    "mouthLowerDown_L", 
    "mouthLowerDown_R", 
    "mouthRollLower", 
    "mouthRollUpper", 
    "mouthPucker", 
    "mouthSmile_L", 
    "mouthSmile_R", 
    "mouthShrugUpper", 
    "mouthShrugLower", 
    "mouthTightener", 
    "mouthUpperUp_L", 
    "mouthUpperUp_R", 
    "noseSneer_L", 
    "noseSneer_R"
  },
  {
    "feature08_jawBConcave", 
    "feature08_jawBBulge", 
    "feature07_jawboneConcave", 
    "feature07_jawboneBulge", 
    "feature06_jawbonePointConcave", 
    "feature06_jawbonePointBulge", 
    "feature06_jawbonePointDownward", 
    "feature06_jawbonePointUpward", 
    "feature05_cheekConcave", 
    "feature05_cheekBulge", 
    "feature04_cheekboneBackward", 
    "feature04_cheekboneDownward", 
    "feature04_cheekboneForward", 
    "feature04_cheekboneUpward", 
    "feature04_cheekboneBulge", 
    "feature04_cheekboneConcave", 
    "feature03_jawBack", 
    "feature03_jawFront", 
    "feature03_jawFlat", 
    "feature03_jawPoint", 
    "feature03_jawShrink", 
    "feature03_jawStretch", 
    "feature02_headThin", 
    "feature02_headFat", 
    "feature02_headShort", 
    "feature02_headLong", 
    "feature01_superciliarisSunken", 
    "feature01_foreHeadBulge", 
    "feature01_foreHeadSunken", 
    "mouth05_mouthBackward", 
    "mouth05_mouthForward", 
    "mouth05_mouthBigger", 
    "mouth05_mouthSmaller", 
    "mouth05_mouthUpward", 
    "mouth05_mouthDownward", 
    "mouth04_mouthCornersUpward", 
    "mouth04_mouthCornersDownward", 
    "mouth04_mouthCornersStretch", 
    "mouth04_mouthCornersShrink", 
    "mouth03_lowerLipWide", 
    "mouth03_lowerLipThin", 
    "mouth03_lowerLipThick", 
    "mouth02_lipAlaDownward", 
    "mouth02_fishPearlBulge", 
    "mouth01_upperLipWiden", 
    "mouth01_upperLipThin", 
    "mouth01_upperLipThick", 
    "nose04_noseAlaBackward", 
    "nose04_noseAlaForward", 
    "nose04_noseAlaLower", 
    "nose04_noseAlaHigher", 
    "nose04_noseAlaNarrow", 
    "nose04_noseAlaWiden", 
    "nose03_noseTipSmaller", 
    "nose03_noseTipBigger", 
    "nose03_noseTipHigher", 
    "nose03_noseTipLower", 
    "nose03_noseTipUpward", 
    "nose03_noseTipDownward", 
    "nose02_noseBridgeNarrow", 
    "nose02_noseBridgeWiden", 
    "nose02_noseBridgeDownward", 
    "nose02_noseBridgeUpward", 
    "nose01_nasionNarrow", 
    "nose01_nasionWiden", 
    "nose01_nasionUpward", 
    "nose01_nasionDownward", 
    "eyes06_eyesBigger", 
    "eyes06_eyesSmaller", 
    "eyes05_outereyeCornerNarrow", 
    "eyes05_outereyeCornerEnlarge", 
    "eyes05_outereyeCornerShrink", 
    "eyes05_outereyeCornerStretch", 
    "eyes04_lowereyelidOuterDownward", 
    "eyes04_lowereyelidInnerDownward", 
    "eyes04_lowereyelidUpward", 
    "eyes04_lowereyelidDownward", 
    "eyes03_uppereyelidInnerUpward", 
    "eyes03_uppereyelidOuterUpward", 
    "eyes03_uppereyelidDownward", 
    "eyes03_uppereyelidUpward", 
    "eyes02_innereyeCornerNarrow", 
    "eyes02_innereyeCornerEnlarge", 
    "eyes02_innereyeCornerShrink", 
    "eyes02_innereyeCornerStretch", 
    "eyes01_eyesDownward", 
    "eyes01_eyesUpward", 
    "eyes01_eyesOutward", 
    "eyes01_eyesInward", 
    "brow04_browOuterInward", 
    "brow04_browOuterOutward", 
    "brow04_browOuterDownward", 
    "brow04_browOuterUpward", 
    "brow03_browMiddleInward", 
    "brow03_browMiddleOutUpward", 
    "brow03_browMiddleDownward", 
    "brow03_browMiddleUpward", 
    "brow02_browInnerInward", 
    "brow02_browInnerOutward", 
    "brow02_browInnerDownward", 
    "brow02_browInnerUpward", 
    "brow01_browInward", 
    "brow01_browOutward", 
    "brow01_browDownward", 
    "brow01_browUpward"
  }
}

local expression_name = "expression"
local identity_name = "identity"
local blendshape_types = {expression_name, identity_name, "other"}

function main:Initialize()
  _COROUTINES_ON();
  
  --self:ExportModels("test:cartoon_male_bs_a_test")
  --self:ExportFaceBs("cartoon_all_bs_g_xiugai");
  --self:ExportModels("test:new_uv_hair", true)
  --self:ExportModels("test:eyebrows")
  --self:ExportModels("test:glasses")
  --self:SendImages("test:face_female")
  --self:ShowResourceUsage("docs:cartoon/resource/default/models")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/female/female_eye_L/data/")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/female/female_eye_R/data/")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/female/female_head_grp/data/")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/male/male_eye_L/data/")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/male/male_eye_R/data/")
  self:ExportMeshToObjs("E:/gerrit/algo_video_facecute/library/test_resource/cartoon/model/male/male_head_grp/data/")
end

local function _DeepCopy(v)
  local t = type(v)
  if t == "table" then
    local c = {}
    for k, d in pairs(v) do
      c[k] = _DeepCopy(d)
    end
    return c
  else
    return v
  end
end

function main:ExportMeshToObjs(dir)
  local files = self:_GetFiles(dir)
  for _, file in ipairs(files) do
    if self:_GetExtension(file) == ".mesh" then
      local file_path = dir..file
      local file_name = self:_GetFileName(file, ".mesh")
      local model = apollonode.MeshNode();
      if not model:CreateResource(file_path, material_path,nil,nil,nil,nil,apolloengine.VertexBufferEntity.MU_DYNAMIC, nil) == false then
        self:_SaveModelToObj(model, file_path..".obj")
      else
        LOG("Faile to load mesh: "..file_path);
      end
    end
  end
end

function main:SendImages(dir)
  local dir_path = venuscore.IFileSystem:PathAssembly(dir.."/");
  local rootconfig = venusjson.LaodJsonFile("test:cartoon/cartoon_load.json")
  
  local weights = {weights = {}}
  local res = web:Initialize(rootconfig.host, rootconfig.port)
  if not res then
    LOG("failed to init web")
    return false
  end
  
  local images = self:_GetFiles(dir_path)
  for i = 1, #images do
    if self:_GetExtension(images[i]) == ".jpg" then
      local res, detail = web:SendImage(dir_path..images[i])
      LOG("finished requesting image "..images[i])
      local file_name = self:_GetFileName(images[i], ".jpg")
      weights.weights[file_name] = detail.id_weights
    end
  end
  
  local result_path = dir_path.."result.json"
  venusjson.SaveJsonObject(weights, result_path)
  return true
end

function main:ShowResourceUsage(root_dir)
  root_dir = venuscore.IFileSystem:PathAssembly(root_dir.."/")
  local config_files = self:_GetFiles(root_dir)
  local data_file_names = {}
  local other_names = {}
  for i = 1, #config_files do
    if self:_GetExtension(config_files[i]) == ".json" then
      local config = venusjson.LaodJsonFile(root_dir..config_files[i])
      local values = self:_GetValues(config)
      for j = 1, #values do
        if values[j]:find("data/", 1, true) then
          local file_path = root_dir..values[j];
          if not venuscore.IFileSystem:isFileExist(file_path) then
            LOG("data file doesn't exist: "..values[j].." configured in "..config_files[i])
          end
          
          local ext_name = self:_GetExtension(file_path)
          local file_name = self:_GetFileName(file_path, ext_name)
          data_file_names[file_name..ext_name] = true
        else
          other_names[values[j]] = true
          
          local file_path = root_dir..values[j];
          if venuscore.IFileSystem:isFileExist(file_path) then
            LOG("file path format is wrong: "..values[j].." configured in "..config_files[i])
          end
        end
      end
    end
  end
  
  local data_dir = root_dir.."data"
  local data_resources = self:_GetFiles(data_dir)
  for i = 1, #data_resources do
    local ext_name = self:_GetExtension(data_resources[i])
    local file_name = self:_GetFileName(data_resources[i], ext_name)
    if not data_file_names[file_name..ext_name] then
      LOG("data resource is not in use: "..file_name..ext_name)
    end
  end  
end

function main:ExportModels(dir, separate_mesh)
  local output_dir = venuscore.IFileSystem:PathAssembly(dir.."/");
  local tex_dir = output_dir.."tex/"
  local config_file = output_dir.."config.json"
  local export_config = venusjson.LaodJsonFile(config_file)
  local venus_dir = output_dir.."venus/";
  local mesh_dir = output_dir.."mesh/";
  local obj_dir = output_dir.."obj/";
  paths.mkdir(venus_dir);
  paths.mkdir(mesh_dir);
  paths.mkdir(obj_dir);
  
  local files = self:_GetFiles(output_dir)
  for i = 1, #files do
    local file_name = nil
    if self:_GetExtension(files[i]) == ".fbx" then
      file_name = self:_GetFileName(files[i], ".fbx")
    elseif self:_GetExtension(files[i]) == ".obj" then
      file_name = self:_GetFileName(files[i], ".obj")
    end
    if file_name ~= nil then
      local fbx_model = output_dir..files[i]
      local venus_output_dir = venus_dir..file_name.."/"
      paths.mkdir(venus_output_dir)
      self:ExportFbx(fbx_model, tex_dir, venus_output_dir)
      
      local mesh_output_dir = mesh_dir..file_name.."/"
      local obj_output_dir = obj_dir..file_name.."/"
      paths.mkdir(mesh_output_dir);
      paths.mkdir(obj_output_dir);
      self:ExportMeshToObj(file_name, venus_output_dir, tex_dir, mesh_output_dir, obj_output_dir, export_config, separate_mesh)
    end
  end
end

function main:ExportFaceBs(name)
  local fbx_dir = "test:"..name
  local fbx_model = venuscore.IFileSystem:PathAssembly(fbx_dir.."/"..name..".fbx");
  local tex_dir = venuscore.IFileSystem:PathAssembly(fbx_dir); 
  local output_dir = venuscore.IFileSystem:PathAssembly(fbx_dir.."/"); 
  local config_file = output_dir.."config.json"
  local export_config = venusjson.LaodJsonFile(config_file)
  
  local venus_dir = output_dir.."venus/";
  local masquerade_dir = output_dir.."masquerade/";
  local obj_dir = output_dir.."obj/";
  local constraint_path = output_dir.."constraint.json"
  
  self:ExportFbx(fbx_model, tex_dir, venus_dir);  
  self:ExportBlendshapeObjs(venus_dir, masquerade_dir, constraint_path)
  self:ExportMeshToObj(name, venus_dir, tex_dir, masquerade_dir, masquerade_dir, export_config, true)
end

function main:ExportFbx(fbx_model, tex_dir, result_dir)
  
  local param = {
    ["input"] = fbx_model,
    ["output"] = result_dir,
    ["texPath"] = tex_dir,
    ["scale"] = 0,
    ["pos"] = 0,
    ["rot"] = 0,
    ["forceMat"] = true,
    ["compress"] = false
  }
  local result = fbxfunc.FbxTool:Convert(param);
end

function main:ExportBlendshapeObjs(venus_dir, result_dir, constraint_path)
  
  local config_path = venus_dir.."config.json";
  local config = venusjson.LaodJsonFile(config_path);
  
  if config.morpher ~= nil and #config.morpher > 0 then
    if config.morpher[1].modellist ~= nil then
      paths.mkdir(result_dir);
      for i = 1, #config.morpher[1].modellist do
        local morpher_model_config = config.morpher[1].modellist[i];
        morpher_model_config.path = string.gsub(morpher_model_config.path, "%.%/", "")     
        local model, result = self:_LoadMorpher(morpher_model_config, venus_dir)
        
        local vertex_data, index_data = self:_GetVerticesAndIndices(model)
        
        local model_obj_dir = result_dir..tostring(i).."/"
        paths.mkdir(model_obj_dir)
        
        local obj_path = model_obj_dir.."base.obj"
        self:_SaveObj(obj_path, vertex_data, index_data)

        local bs_info = {}
        for j = 1, #blendshape_types do
          bs_info[j] = {}
          bs_info[j].bs_obj_dir = model_obj_dir..blendshape_types[j].."/"
          paths.mkdir(bs_info[j].bs_obj_dir)
          --bs_info[j].morpher_config = self:_GetMorpherModelConfig(config.morpher[1], i)
          --bs_info[j].channel_names = {}
        end
        
        local group_cnt = result.model:GetNumberOfGroup()
        for l = 1, group_cnt do
          local group_index = l-1
          local channel_cnt = result.model:GetNumberOfChannel(group_index)
          for j = 1, channel_cnt do
            local channel_name = config.morpher[1].modellist[i].group[l].channels[j].name;
            local channel_vertex_data = result.model:GetChannelVertices(group_index, j-1, 0);
            for k = 1, #channel_vertex_data do
              channel_vertex_data[k][1] = channel_vertex_data[k][1] + vertex_data[k][1]
              channel_vertex_data[k][2] = channel_vertex_data[k][2] + vertex_data[k][2]
              channel_vertex_data[k][3] = channel_vertex_data[k][3] + vertex_data[k][3]
            end
            
            local bs_name = self:_FindBlendshapeName(channel_name)
            --morpher_model_config.group[l].channels[j].name = bs_name.bs_name
            --table.insert(bs_info[bs_name.ret].morpher_config.morpher[1].modellist[1].group[l].channels, morpher_model_config.group[l].channels[j])
            --table.insert(bs_info[bs_name.ret].channel_names, bs_name.bs_name)
            local channel_obj_path = bs_info[bs_name.ret].bs_obj_dir..bs_name.bs_name..".obj"          
            self:_SaveObj(channel_obj_path, channel_vertex_data, index_data)
          end
        end
        
        --[[
        for j = 1, #bs_info do
          local morpher_config_path = bs_info[j].bs_obj_dir..blendshape_types[j]..".json"
          venusjson.SaveJsonObject(bs_info[j].morpher_config, morpher_config_path)
          local channel_name_path = bs_info[j].bs_obj_dir.."bs_names.json"
          venusjson.SaveJsonObject(bs_info[j].channel_names, channel_name_path)
        end]]
        
        --self:_SaveBlendshapes(model_obj_dir, venus_dir, constraint_path)
      end
    end    
  end
end

function main:ExportMeshToObj(fbx_name, venus_dir, tex_dir, mesh_result_dir, obj_result_dir, export_config, separate_mesh)
  local venus_config_path = venus_dir.."config.json";
  local config = venusjson.LaodJsonFile(venus_config_path);
  local tex_map = export_config.tex_map
  local default_config = export_config.default
  local fixed_textures = export_config.fixed_textures
  local mesh_json_key = export_config.mesh_json_key
  
  local mesh_list = {}      
  local fbx_config = {models = {}}
  
  if config.meshes ~= nil then
    for i = 1, #config.meshes do
      local mesh_config, mesh_path, mesh_name, texture_files = self:_GetMeshConfig(config.meshes[i], export_config)
      local current_mesh_dir = mesh_result_dir
      if separate_mesh == true then
        current_mesh_dir = mesh_result_dir..mesh_name.."/"
      end
      
      self:_GenerateMesh(venus_dir..mesh_path, tex_dir, texture_files, mesh_name, current_mesh_dir, obj_result_dir)

      table.insert(fbx_config.models, mesh_config)
      if separate_mesh == true then
        venusjson.SaveJsonObject(fbx_config, current_mesh_dir..mesh_name..".json")
        fbx_config.models = {}
      end
      
      table.insert(mesh_list, mesh_name)
    end
    
    if not separate_mesh then
      venusjson.SaveJsonObject(fbx_config, mesh_result_dir..fbx_name..".json")
    end
    venusjson.SaveJsonObject(mesh_list, obj_result_dir.."mesh_list.json")
  end  
  
  return true
end

function main:_GetMorpherModelConfig(morpher_config, model_index)
  local result_config = {morpher = {}}
  result_config.morpher[1] = {
    clipstart = morpher_config.clipstart,
    clipend = morpher_config.clipend,
    modellist = {}}
  result_config.morpher[1].modellist[1] = {
    name = morpher_config.modellist[model_index].name,
    path = morpher_config.modellist[model_index].path,
    channels = {}}
  return result_config
end

function main:_GetVerticesAndIndices(model)
  local vertex_stream = model:GetVertexStream()
  local vertex_data = vertex_stream:GetVertexData()
  local index_stream = model:GetIndexStream()
  local index_data = index_stream:GetIndicesData()
  return vertex_data, index_data
end

function main:_SaveBlendshapes(config_root, mesh_root, constraint_path)
  if constraint_path == nil or not venuscore.IFileSystem:isFileExist(constraint_path) then
    LOG("constraint_path not exist "..constraint_path)
    return
  end
  local id_config = venusjson.LaodJsonFile(config_root..identity_name.."/"..identity_name..".json")
  if id_config.morpher[1].modellist == nil or #id_config.morpher[1].modellist == 0 then
    LOG("id morpher doesn't exist in "..config_root)
    return
  end
  local id_morpher_config = id_config.morpher[1].modellist[1]
  local id_model, id_morpher_result = self:_LoadMorpher(id_morpher_config, mesh_root)
  local modeling_bs_names = {names = id_morpher_result.names}
  venusjson.SaveJsonObject(modeling_bs_names, config_root..identity_name.."/modeling_bs_names.json")
  
  local expr_config = venusjson.LaodJsonFile(config_root..expression_name.."/"..expression_name..".json")
  if expr_config.morpher[1].modellist == nil or #expr_config.morpher[1].modellist == 0 then
    LOG("expr morpher doesn't exist in "..config_root)
    return
  end
  local expr_morpher_config = expr_config.morpher[1].modellist[1]
  local expr_model, expr_morpher_result = self:_LoadMorpher(expr_morpher_config, mesh_root)
  local transfer_bs_names = {names = expr_morpher_result.names}
  venusjson.SaveJsonObject(transfer_bs_names, config_root..expression_name.."/transfer_bs_names.json")

  local face_transfer = libmasquerade.FaceTransfer()
  local constraint_config = venusjson.LaodJsonFile(constraint_path);
  if face_transfer:Load(id_morpher_result.model, expr_morpher_result.model, constraint_config) then
    face_transfer:Save(config_root..identity_name.."/modeling_bs.bin", config_root..expression_name.."/transfer_bs.bin")
  end
  LOG("failed to load face transfer")
end

function main:_GenerateMesh(venus_mesh_path, tex_dir, texture_files, mesh_name, mesh_result_dir, obj_result_dir)
  local data_dir = mesh_result_dir.."data"
  paths.mkdir(data_dir)
  
  local model = apollonode.MeshNode();
  if model:CreateResource(venus_mesh_path, material_path,nil,nil,nil,nil,apolloengine.VertexBufferEntity.MU_DYNAMIC, nil) == false then
    LOG("Faile to load mesh: "..venus_mesh_path);
    return false;
  end
  self:_SaveModelToObj(model, obj_result_dir..mesh_name..".obj")
  self:_CopyFile(venus_mesh_path, data_dir)
  
  for i = 1, #texture_files do
    self:_CopyFile(tex_dir..texture_files[i], data_dir)
  end
end

function main:_LoadMorpher(morpher_model_config, mesh_root)
  local model = apollonode.MeshNode();
  local mesh_path = mesh_root..morpher_model_config.path
  if model:CreateResource(mesh_path, material_path,nil,nil,nil,nil,apolloengine.VertexBufferEntity.MU_DYNAMIC, nil) == false then
    LOG("Faile to load mesh: "..mesh_path);
    return nil;
  end
  return model, utility.CreateMorpherComponent(morpher_model_config, model, mesh_root);
end

function main:_GetMeshConfig(venus_config, export_config)
  local tex_map = export_config.tex_map
  local default_config = export_config.default
  local mesh_json_key = export_config.mesh_json_key
  local fixed_textures = export_config.fixed_textures
  
  local mesh_path = string.gsub(venus_config.name, "%.%/", "")
  local mesh_name = self:_GetFileName(mesh_path, ".mesh")

  local mesh_config = _DeepCopy(default_config)
  mesh_config.mesh = mesh_path
  if mesh_json_key ~= nil then
    mesh_config[mesh_json_key] = "data/"..mesh_name..".json"
  end
  
  local textures = nil
  if tex_map == nil then
    textures = self:_GetTextures(venus_config)
  else
    textures = self:_GetTextureInMap(mesh_name, tex_map)
  end
  
  local texture_files = {}

  if textures.color ~= nil then
    mesh_config.texture = "data/"..textures.color
    table.insert(texture_files, textures.color)
  end
  if textures.normal ~= nil then
    mesh_config.normal = "data/"..textures.normal
    table.insert(texture_files, textures.normal)
  end
  if textures.specular ~= nil then
    mesh_config.specular = "data/"..textures.specular
    table.insert(texture_files, textures.specular)
  end
  if textures.gloss ~= nil then
    mesh_config.gloss = "data/"..textures.gloss
    table.insert(texture_files, textures.gloss)
  end
  
  if fixed_textures ~= nil then
    for j = 1, #fixed_textures do
      table.insert(texture_files, fixed_textures[j])
    end
  end
  
  return mesh_config, mesh_path, mesh_name, texture_files
end

function main:_SaveModelToObj(model, obj_path)
  local vertex_stream = model:GetVertexStream()
  local vertex_data = vertex_stream:GetVertexData()
  local index_stream = model:GetIndexStream()
  local index_data = index_stream:GetIndicesData()

  self:_SaveObj(obj_path, vertex_data, index_data)
end

function main:_SaveObj(obj_path, vertex_data, index_data)
  local file, err = io.open(obj_path, "w")
  if err ~= nil then
    LOG("open file "..obj_path.." failed as "..err)
    return false
  end

  for i = 1, #vertex_data do
    file:write("v "..vertex_data[i][1].." "..vertex_data[i][2].." "..vertex_data[i][3], "\n")
  end
  
  if index_data ~= nil then
    for i = 1, #index_data, 3 do
      file:write("f "..(index_data[i]+1).." "..(index_data[i+1]+1).." "..(index_data[i+2]+1), "\n")
    end
  end
  file:close()
  return true
end

function main:_FindBlendshapeName(name)
  local result = {bs_name = name, ret = 3}  
  local max_len = 0
  for i = 1, #blendshape_names do
    for j = 1, #blendshape_names[i] do
      local lower = blendshape_names[i][j]:lower()
      if name:find(lower, 1, true) ~= nil then
        local len = string.len(lower)
        if len > max_len then
          max_len = len
          result.bs_name = blendshape_names[i][j]
          result.ret = i
        end
      end
    end
  end
  return result
end

function main:_GetFileName(file, ext)
  local file_name = file:match("[^/]*"..ext.."$")
  return file_name:sub(0, #file_name - #ext)
end

function main:_GetTextures(mesh_config)
  local result = {}
  if mesh_config.material ~= nil and
     mesh_config.material.pbrmetallicroughness ~= nil then
    if mesh_config.material.pbrmetallicroughness.basecolortexture ~= nil then
      result.color = string.gsub(mesh_config.material.pbrmetallicroughness.basecolortexture.path, "%.%/data/", "")
    end
  end
  return result
end

function main:_GetTextureInMap(mesh_name, tex_map)
  local result = {}
  if tex_map ~= nil and tex_map[mesh_name] ~= nil then
    local tex_name = tex_map[mesh_name]
    result.color = tex_name..".tga"
    result.normal = tex_name.."_n.tga"
    result.specular = tex_name.."_sp.tga"
    result.gloss = tex_name.."_g.tga"
  end
  return result
end

function main:_CopyFile(file_path, target_dir)
  local cmd = "copy "..file_path.." "..target_dir
  cmd = string.gsub(cmd, "%/", "\\")
  os.execute(cmd)
  LOG(cmd)
end

function main:_GetFiles(directory)
  local i, t, popen = 0, {}, io.popen
  for filename in popen('dir "'..directory..'" /b'):lines() do
      i = i + 1
      t[i] = filename
  end
  return t
end

function main:_GetExtension(filename)
    return filename:match("^.+(%..+)$")
end

function main:_GetValues(root)
  local res = {}
  local t = type(root)
  if t == "table" then
    for _, child in pairs(root) do
      local sub = self:_GetValues(child)
      for _, v in pairs(sub) do 
          table.insert(res, v)
      end
    end    
  else
    table.insert(res, tostring(root))
  end
  return res
end

return main;