require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.ellipselen"

local MouthScale = {}

function MouthScale:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function MouthScale:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function MouthScale:_InitParams(defined)
  beautydefined = defined
  --嘴型参数
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_MOUTHCOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_MOUTHCOEF");
  apolloengine.ShaderEntity.MOUTHPOINTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"MOUTHPOINTS");
  apolloengine.ShaderEntity.MOUTHPARAMS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"MOUTHPARAMS");
  apolloengine.ShaderEntity.MOUTHPARAMSMINUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"MOUTHPARAMSMINUS");
end

function MouthScale:SetParams(config, shaderParams)
  shaderParams.mouthpointArray = mathfunction.vector2array();
  shaderParams.mouthParaArray = mathfunction.vector2array();
  shaderParams.mouthParaMinusArray = mathfunction.vector2array();

  shaderParams.mouthParaArray:PushBack(beautyutils._ArrayToVec2(config.MouthRadius));
  shaderParams.mouthParaArray:PushBack(mathfunction.vector2(config.MouthScale[1], config.MouthScale[2]));

  if(config.MouthScale_minus ~= nil) then
    shaderParams.mouthParaMinusArray:PushBack(beautyutils._ArrayToVec2(config.MouthRadius));
    shaderParams.mouthParaMinusArray:PushBack(mathfunction.vector2(config.MouthScale_minus[1], config.MouthScale_minus[2]));
  else
    shaderParams.mouthParaMinusArray:PushBack(beautyutils._ArrayToVec2(config.MouthRadius));
    shaderParams.mouthParaMinusArray:PushBack(mathfunction.vector2(-config.MouthScale[1], -config.MouthScale[2]));
  end
  self:_InitLens(config)
end

function MouthScale:_InitLens(config)
  self.lens = {}
  self.lens[#self.lens + 1] = Len(beautydefined.MouthIndices, config.MouthScale, config.MouthRadius)
end

function MouthScale:_InitQuad(quadnode, shaderParams)
  -- local mouthTriangles = mathfunction.vector3array();
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_MOUTHCOEF, mathfunction.vector1(0.0));
  quadnode:SetParameter(apolloengine.ShaderEntity.MOUTHPARAMS, shaderParams.mouthParaArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.MOUTHPOINTS, shaderParams.mouthpointArray);

  quadnode:SetParameter(apolloengine.ShaderEntity.MOUTHPARAMSMINUS, shaderParams.mouthParaMinusArray);
end

function MouthScale:_DrawQuad(quadnode, keypoints, rate)
  local mouthpointArray = mathfunction.vector2array();
  -- LOG("keypoints size: " .. #keypoints);
  -- LOG("Output mouth points: ");
  -- for i = 1, #beautydefined.MouthIndices, 1 do
  --   LOG(keypoints[beautydefined.MouthIndices[i] + 1][1] .. " " .. keypoints[beautydefined.MouthIndices[i] + 1][2])
  -- end
  -- LOG("mouth points end");

  for i = 1, #beautydefined.MouthIndices, 1 do
    mouthpointArray:PushBack(beautyutils._ArrayToVec2(keypoints[beautydefined.MouthIndices[i] + 1]));
  end
  
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_MOUTHCOEF, mathfunction.vector1(self.coef * rate));
  quadnode:SetParameter(apolloengine.ShaderEntity.MOUTHPOINTS, mouthpointArray);
end

function MouthScale:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef * rate
  local dy_add = 0
  if config.LipParam then
    dy_add = beautyutils.GetDistance(keypoints[beautydefined.MouthUpOuterMid + 1], keypoints[beautydefined.MouthUpInnerMid + 1])*config.LipParam[2]*self.coef*rate;
  end
  for i = 1, #self.lens do
    local cornerIdx = self.lens[i]:GetCornerIdx()
    local kp1 = keypoints[cornerIdx[1] + 1];
    local kp2 = keypoints[cornerIdx[2] + 1];
    local kp3 = keypoints[cornerIdx[3] + 1];
    local kp4 = keypoints[cornerIdx[4] + 1];

    self.lens[i]:InitMouthParams(kp1, kp2, kp3, kp4, facedir, coef, dy_add) 
  end
  
  local indexRange = self:_getIndexRange(#keypoints)
  for i = 1, #indexRange do
    local index = indexRange[i] + 1
    for j = 1, #self.lens do
      local tmp = self.lens[j]:WarpObliqueEllipseScale(keypoints[index], aspect_ratio)
      keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmp[1]
      keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmp[2]
    end
  end
end

function MouthScale:_getIndexRange(num)
  local min = 84
  local max = 103
  local range = {}
  for i = min, max do
    table.insert(range, i)
  end
  
  if num >= 240 then
    min = 70
    max = 133
    for i = min, max do
      table.insert(range, i + 106)
    end
  end
  return range
end

function MouthScale:UpdateParaValue(quadnode, config, key_index, attribute, value)
--  LOG("mouth update: " .. attribute .. " " .. value);
  if(attribute == "Inten") then
    config.MouthScale[1] = value;
--    LOG("mouth_scale_x set to: " .. config.MouthScale[1])
  end
  if (attribute == "YScale") then
    config.MouthScale[2] = value;
  end
  if(attribute == "RR") then
    config.MouthRadius[1] = value;
    -- config.MouthRadius[2] = value * 0.4;
--    LOG("mouth_RR set to: " .. config.MouthRadius[1]);
  end
  if (attribute == "rr") then
    config.MouthRadius[2] = value;
--    LOG("mouth_rr set to: " .. config.MouthRadius[2])
--    LOG("mouth_rr set disabled");
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.MOUTHPARAMS, shaderParams.mouthParaArray);
end

function MouthScale:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0;
  end
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_MOUTHCOEF,mathfunction.vector1(self.coef * rate));
  --LOG("Mouth Scale coef set to: " .. self.coef);
end

function MouthScale:GetParams(config)
  return {config.MouthRadius[1], config.MouthRadius[2], config.MouthScale[1], config.MouthScale[2]};
end

function MouthScale:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  if(self.coef < 0) then
    self.coef = coef * 0.7;
  else
    self.coef = coef * 0.6;
  end
  --WARNING("[facelift]: Mouth Scale coef: " .. self.coef)
end

function MouthScale:Release()
  self.lens = {}
end
  
return MouthScale