require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local HairLine = {}

function HairLine:_Bbox(config, keypoints, bboxData)
  if #keypoints < 129 then
    return ;
  end
  local hairline_indices = beautydefined.HairlineIndices;
  local hairline_directions = beautydefined.HairlineDir;
	for i = 1, #hairline_indices do
    local src = keypoints[hairline_indices[i][1] + 1];
    local dst = keypoints[hairline_directions[i][1] + 1];
    local dr = config.ForeHeadRadius[i][1] * beautyutils.GetDistance(src, dst)
    bboxData.min_x = math.min(bboxData.min_x, src[1]-dr);
    bboxData.max_x = math.max(bboxData.max_x, src[1]+dr);
    bboxData.min_y = math.min(bboxData.min_y, src[2]-dr);
    bboxData.max_y = math.max(bboxData.max_y, src[2]+dr);
    
    src = keypoints[hairline_indices[i][2] + 1];
    dst = keypoints[hairline_directions[i][2] + 1];
    dr = config.ForeHeadRadius[i][1] * beautyutils.GetDistance(src, dst)
    bboxData.min_x = math.min(bboxData.min_x, src[1]-dr);
    bboxData.max_x = math.max(bboxData.max_x, src[1]+dr);
    bboxData.min_y = math.min(bboxData.min_y, src[2]-dr);
    bboxData.max_y = math.max(bboxData.max_y, src[2]+dr);
  end
end

function HairLine:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function HairLine:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function HairLine:_InitParams(defined)
    beautydefined = defined
    --Hairline pulling parameters
    self.coef = self.coef or 0
    self.coef_temp = 1.0;
    --shader params
    apolloengine.ShaderEntity.UNIFORM_HAIRLINECOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "UNIFORM_HAIRLINECOEF");
    apolloengine.ShaderEntity.HAIRLINEPARA = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "HAIRLINEPARA");
    apolloengine.ShaderEntity.HAIRLINEPARARADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "HAIRLINEPARARADIUS");
    apolloengine.ShaderEntity.HAIRLINEPOINTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "HAIRLINEPOINTS");
end

function HairLine:SetParams(config, shaderParams)
    shaderParams.hairlinePointArray = mathfunction.vector4array();
    shaderParams.hairlineParamArray = mathfunction.vector2array();
    shaderParams.hairlineParamRadiusArray = mathfunction.vector2array();

    for i = 1, #beautydefined.HairlineIndices - 1 do
        shaderParams.hairlineParamArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadParam[i]));
        shaderParams.hairlineParamArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadParam[i]));
        shaderParams.hairlineParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadRadius[i]));
        shaderParams.hairlineParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadRadius[i]));
    end

    shaderParams.hairlineParamArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadParam[#beautydefined.HairlineIndices]));
    shaderParams.hairlineParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.ForeHeadRadius[#beautydefined.HairlineIndices]));
    
    self:_InitLens(config)
end

function HairLine:_InitLens(config)
  self.fastlens = {}
  for i = 1, #beautydefined.HairlineIndices - 1 do
    local index1 = beautydefined.HairlineIndices[i][1] + 1
    local index2 = beautydefined.HairlineDir[i][1] + 1;
    local scale = beautydefined.ForeheadScaleScript[i]
    self.fastlens[#self.fastlens + 1] = Len(index1, index2, beautydefined.ForeheadParamScript[i][1] * scale, beautydefined.ForeheadParamScript[i][2] * scale, beautydefined.ForeheadParamRadiusScript[i])
    index1 = beautydefined.HairlineIndices[i][2] + 1
    index2 = beautydefined.HairlineDir[i][2] + 1;
    self.fastlens[#self.fastlens + 1] = Len(index1, index2, beautydefined.ForeheadParamScript[i][1] * scale, beautydefined.ForeheadParamScript[i][2] * scale, beautydefined.ForeheadParamRadiusScript[i])
  end
  local index1 = beautydefined.HairlineIndices[#beautydefined.HairlineIndices][1] + 1
  local index2 = beautydefined.HairlineDir[#beautydefined.HairlineIndices][1] + 1;
  local scale = beautydefined.ForeheadScaleScript[#beautydefined.HairlineIndices]
  self.fastlens[#self.fastlens + 1] = Len(index1, index2, 
    beautydefined.ForeheadParamScript[#beautydefined.HairlineIndices][1] * scale, 
    beautydefined.ForeheadParamScript[#beautydefined.HairlineIndices][2] * scale, 
    beautydefined.ForeheadParamRadiusScript[#beautydefined.HairlineIndices])
end

function HairLine:_InitQuad(quadnode, shaderParams)
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_HAIRLINECOEF, mathfunction.vector1(0.0));
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPARA, shaderParams.hairlineParamArray);
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPARARADIUS, shaderParams.hairlineParamRadiusArray);
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPOINTS, shaderParams.hairlinePointArray);
end

function HairLine:_DrawQuad(quadnode, keypoints, rate)
    -- Hair line points
    local hairline_indices = beautydefined.HairlineIndices;
    local hairline_directions = beautydefined.HairlineDir;
    local hairline_points = mathfunction.vector4array();
    for i = 1, #hairline_indices - 1 do
      if #keypoints >= 129 then
        local src = keypoints[hairline_indices[i][1] + 1];
        local dst = keypoints[hairline_directions[i][1] + 1];
        hairline_points:PushBack(beautyutils._ArrayToVec4(src, dst));

        src = keypoints[hairline_indices[i][2] + 1];
        dst = keypoints[hairline_directions[i][2] + 1];
        hairline_points:PushBack(beautyutils._ArrayToVec4(src, dst));
      else
        hairline_points:PushBack(beautyutils._ArrayToVec4({0, 0}, {0, 0}));
        hairline_points:PushBack(beautyutils._ArrayToVec4({0, 0}, {0, 0}));
      end
    end
    
    local coef = 0
    if #keypoints >= 129 then
      coef = self.coef
      hairline_points:PushBack(beautyutils._ArrayToVec4(keypoints[hairline_indices[#hairline_indices][1]+1], keypoints[hairline_directions[#hairline_directions][1]+1]));
    else
      hairline_points:PushBack(beautyutils._ArrayToVec4({0, 0}, {0, 0}));
    end
    
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_HAIRLINECOEF, mathfunction.vector1(coef * rate));
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPOINTS, hairline_points);
end

function HairLine:GetParams(config)
    local hairlineParam = beautyutils.Clone(config.ForeHeadParam);
    local hairlineParamRadius = beautyutils.Clone(config.ForeHeadRadius);
    return hairlineParam, hairlineParamRadius
end

function HairLine:UpdateParaValue(quadnode, config, key_index, attribute, value)
    if(attribute == "X") then
        config.ForeHeadParam[key_index - beautydefined.HairLine_start + 1][1] = value;
    end
    if(attribute == "Y") then
        config.ForeHeadParam[key_index - beautydefined.HairLine_start + 1][2] = value;
    end
    if(attribute == "R") then
        config.ForeHeadRadius[key_index - beautydefined.HairLine_start + 1][1] = value;
    end

    local shaderParams = {}
    self:SetParams(config, shaderParams)
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPARA, shaderParams.hairlineParamArray);
    quadnode:SetParameter(apolloengine.ShaderEntity.HAIRLINEPARARADIUS, shaderParams.hairlineParamRadiusArray);
end

function HairLine:UpdateCoeff(quadnode, value, rate)
    if (value == 1) then
        self.coef = 1.0;
    else
        self.coef = 0.0;
    end
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_HAIRLINECOEF, mathfunction.vector1(self.coef * rate));
end

function HairLine:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Chin Scale coef: " .. self.coef)
end

function HairLine:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  if #keypoints < 263 and #keypoints >= 240 then
    return;
  end
  
  if #keypoints < 129 and #keypoints >= 106 then
    return;
  end
  
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local shift = 0
  if #keypoints >= 240 then
    shift = 134
  end
  local coef = self.coef * rate
  local tmpDis = {}
  for i = 1, 23 do
    tmpDis[i] = {0, 0}
  end
  for i = 1, #self.fastlens do
    local srcIdx = self.fastlens[i]:GetSrcIdx() + shift
    local dstIdx = self.fastlens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.fastlens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)
    local tmp = self.fastlens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx - shift - 106] = tmp
  end
  
  beautyutils.LinearInterpolation(tmpDis, 4, 20)
  for i = 1, #tmpDis do
    local index = i + shift + 106
    keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmpDis[i][1]
    keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmpDis[i][2]
  end
end

function HairLine:_getIndexRange(num)
  local range = {}

  if num >= 263 then
    for i = 2, 20 do
      table.insert(range, i + 240)
    end
  else
    if num >= 129 then
      for i = 2, 20 do
        table.insert(range, i + 106)
      end
    end
  end
  
  return range
end

function HairLine:Release()
  self.fastlens = {}
end

return HairLine