local venuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"
local cv = require "computervisionfunction"
local mf = require "mathfunction"

local AE = require "apolloengine"
local BD = require "bluecore.bluedefined"
local BlueNode = require "bluecore.bluenode"
local BlueSelf = require "bluecore.blueself"
local EventDispatcher = require "bluecore.eventdispatcher"
local BluePinInfo = require "bluecore.bluepininfo"

local HandIndex = BlueNode:extend();

---- 游戏运行模式
function HandIndex:new(...)
  HandIndex.super.new(self, ...)
  self.output = {
     humanId = 0,
     point = mf.vector2(0,0),
     size = mf.vector2(0,0),
     indexPoint = mf.vector2(0,0),
     isHigh = false,
     isEdge = false,
     isFingerUp = false,
     curType = 0,
     numHand = 0
  }
  self.target = {
    humanId = 0,
    point = mf.vector2(0,0),
    size = mf.vector2(0,0),
    indexPoint = mf.vector2(0,0),
    isHigh = false,
    isEdge = false,
    isFingerUp = false,
    curType = 0,
    numHand = 0
  }
  self.finger = {
    humanId = 0,
    point = mf.vector2(0,0),
    size = mf.vector2(0,0),
    indexPoint = mf.vector2(0,0),
    isHigh = false,
    isEdge = false,
    isFingerUp = false,
    curType = 0,
    numHand = 0
  }

  self.isCfgEdge = false ;

  self.missedFrames = 0;
  self.missedThre = 10;
  self.meetEdge = false;
  self.lastDx = 0;
  self.lastDy = 0;

  self.lastFinger = {
    humanId = 0,
    point = mf.vector2(0,0),
    size = mf.vector2(0,0),
    indexPoint = mf.vector2(0,0),
    isHigh = false,
    isEdge = false,
    isFingerUp = false,
    scale = 0
  };
  self.lastResult = {
    humanId = 0,
    point = mf.vector2(0,0),
    size = mf.vector2(0,0),
    indexPoint = mf.vector2(0,0),
    isHigh = false,
    isEdge = false,
    isFingerUp = false,
    scale = 0
  };

  self.lastId = 0;
end


function iou(hand1, hand2)
  local x1 = math.max(hand1.point.mx - hand1.size.mx / 2.0, hand2.point.mx - hand2.size.mx / 2.0)
  local y1 = math.max(hand1.point.my - hand1.size.my / 2.0, hand2.point.my - hand2.size.my / 2.0)
  local x2 = math.min(hand1.point.mx + hand1.size.mx / 2.0, hand2.point.mx + hand2.size.mx / 2.0)
  local y2 = math.min(hand1.point.my + hand1.size.my / 2.0, hand2.point.my + hand2.size.my / 2.0)
  if x2 - x1 < 0 or y2 - y1 < 0 then
    return 0;
  end
  local area1 = hand1.size.mx * hand2.size.my;
  local area2 = hand2.size.mx * hand2.size.my;
  local inter = (x2 - x1) * (y2 - y1);
  return inter / (area1 + area2 - inter);
end

function match(hand1, hand2)
  local v = iou(hand1, hand2);
  return v > 0.5;
end

function HandIndex:_OnUpdate(eventParams)
  self.output = eventParams ;

  -- init
  if self.output.humanId == 1 then
    self.target = {
      humanId = 0,
      point = mf.vector2(0,0),
      size = mf.vector2(0,0),
      indexPoint = mf.vector2(0,0),
      isHigh = false,
      isEdge = false,
      isFingerUp = false,
      curType = 0,
      numHand = 0
    };
    self.finger = {
      humanId = 0,
      point = mf.vector2(0,0),
      size = mf.vector2(0,0),
      indexPoint = mf.vector2(0,0),
      isHigh = false,
      isEdge = false,
      isFingerUp = false,
      curType = 0,
      numHand = 0
    };
  end


  -- if self.output.humanId == self.lastResult.humanId then
  if self.lastFinger.humanId ~= nil and self.lastFinger.humanId ~= 0 and match(self.output, self.lastFinger) then
    self.target = self.output;
  end
  if self.finger.humanId == 0 and self.output.curType == 1048576 then
    self.finger = self.output;
  end

  if self.output.humanId == self.output.numHand then
    if self.finger.humanId ~= 0 then
      self.output = self.finger;
    elseif self.target.humanId ~= 0 then
      self.output = self.target;
    else

    end

    self.lastFinger = self.output;

    local resolution
    if _KRATOSEDITOR then
      local scene = AE.SceneManager:GetOrCreateScene("facecuteScene"); --previewscene
      local fbo = scene:GetDefaultRenderTarget();
      local tex = fbo:GetAttachment(AE.RenderTargetEntity.TA_COLOR_0);
      resolution = tex:GetSize();
    else
      resolution = AE.Framework:GetResolution();
    end
    local imgRatio = resolution:x()/resolution:y();
    local distance = self.inputArgs[5];
    local renderScale = self.inputArgs[6];

    local cx = 2.0 * self.output.point.mx + self.output.size.mx - 1.0;
    local cy = -(2.0 * self.output.point.my + self.output.size.my - 1.0);
    cx = 1.0 * cx * imgRatio;

    local w = 2.0 * self.output.size.mx * imgRatio;
    local h = 2.0 * self.output.size.my;

    local px = 2.0 * self.output.indexPoint.mx - 1.0;
    local py = -(2.0 * self.output.indexPoint.my - 1.0);
    px = 1.0 * px * imgRatio;

    -- not index finger
    local tmpx = self.lastDx;
    local tmpy = self.lastDy;
    if self.output.curType ~= 1048576 then
      px = cx + self.lastDx;
      py = cy + self.lastDy;
    end

    self.lastDx = px - cx;
    self.lastDy = py - cy;

--    print(self.output.humanId, self.lastId, self.output.numHand, self.output.curType, self.lastDx, self.lastDy)

    self.lastId = self.output.humanId;

    local scale = math.sqrt((px - cx) * (px - cx) + (py - cy) * (py - cy)) * renderScale

    -- not index finger
    if self.output.curType ~= 1048576 then
      scale = self.lastResult.scale;
      self.missedFrames = self.missedFrames + 1;
    else
      self.missedFrames = 0;
    end

    if self.lastResult.humanId == self.output.humanId and self.lastResult.curType == self.output.curType and math.abs(self.lastResult.scale - scale) > 0.3 then
      scale = self.lastResult.scale;
      px = cx + tmpx;
      py = cy + tmpy;
    end
--    print("scale ", scale, self.lastResult.scale);

    if self.missedFrames < self.missedThre then
      self.output.isHigh = true;
    else
      self.output.isHigh = false;
    end

    if self.missedFrames == self.missedThre then
      self.output.isEdge = true;
    end

    py = py + scale * distance


    self.lastResult = {
      humanId = self.output.humanId,
      curType = self.output.curType,
      point = mf.vector2(cx, cy),
      size = mf.vector2(w, h),
      indexPoint = mf.vector2(px, py),
      isHigh = self.output.isHigh,
      isEdge = self.output.isEdge,
      isFingerUp = self.output.isFingerUp,
      scale = scale
    };
  end

  return self.lastResult.humanId, self.lastResult.point, self.lastResult.size, self.lastResult.indexPoint,
  self.lastResult.isHigh, self.lastResult.isEdge, self.lastResult.isFingerUp, self.lastResult.scale;

--  return self.output.humanId, mf.vector2(cx, cy), mf.vector2(w, h), mf.vector2(px, py), self.output.isHigh, self.output.isEdge,
--    self.output.isFingerUp, scale
--  if self.output.humanId < self.output.numHand then
--    return self.lastResult;
--  else
--    return self.lastResult;
--  end

end

function HandIndex:_OnNextBranch()
  if self.isCfgEdge and not self.output.isEdge then
    return 0
  end

  if self.output.isHigh then
    return 1
  else
    return 2
  end
end

function HandIndex:RegisterTriggerFunc(func, bluePrintSelf)
  self:_GetInput()

  local type = self.inputArgs[2];

  self.isCfgEdge = self.inputArgs[3];

  WARNING("HandIndex " .. tostring(type) );

--  EventDispatcher:RegisterCallback(self.inputArgs[1]:GetContentPath(), type, func, bluePrintSelf)
  for k,v in pairs(cv.ClassifyComponent:GetHandTypes()) do
    EventDispatcher:RegisterCallback(self.inputArgs[1]:GetContentPath(), k, func, bluePrintSelf)
  end
end


function HandIndex:_OnCompile()
  self.compileResult = nil
  if HandIndex.super._OnCompile(self) then
    if #self.inputs[1].links ~= 0 then
      local bluePinLink = self.inputs[1].links[1] -- 输入引脚只允许一个连接
      local otherNode = self.graph:GetNode(bluePinLink.nodeUid);

      -- 确保上一个节点是 引用Component
      if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp == nil then
        self.compileResult = BD.EHINT.NOT_LINK_TO_COMP
        return false
      end

      if venuscore.isNil(otherNode.comp) then
        self.compileResult = BD.EHINT.COMP_LOST
        return false
      end

      -- 节点配置值(引脚字面值)不能超过当前反射面板配置值 在编译时(和当前蓝图显示时)自动去除超出范围的
      local configs, hints = self:_GetCompCurrentDetectType(otherNode.comp);
      local config =   self.inputs[2].literal
      local found = false ;
      for _, cfg in pairs(configs) do
        if config == cfg then
          found = true
        end
      end
      if not found then
        self.inputs[2].literal = 0 -- 存放节点配置值(会序列化)
      end

      self.myUnFixComboType:SetData(configs, hints);
    else
      self:_Reset();
      self.compileResult = BD.EHINT.PIN_1_EMPTY;
    end
    return true ;
  end
  return false
end


function HandIndex:_Reset()
  -- 引脚类型
  self.myUnFixComboType:SetData({},{});
  -- 字面值
  self.inputs[2].literal = 0
end

function HandIndex:_OnDeserialize()
  HandIndex.super._OnDeserialize(self);
  self:_SetupInstanceInfoTable();
end

function HandIndex:_SetupInstanceInfoTable()

  -- override Recognition infoTable
  local staticExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  local staticExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]
  local staticDataOutputsInfo = self.infoTable[BD.PIN_DATA_OUTPUT]
  local staticDataInputsInfo = self.infoTable[BD.PIN_DATA_INPUT]
  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = {}
  self.infoTable[BD.PIN_DATA_OUTPUT] = staticDataOutputsInfo -- 并不会动态修改 data output info
  self.infoTable[BD.PIN_EXEC_INPUT]  = staticExecInputsInfo
  self.infoTable[BD.PIN_EXEC_OUTPUT] = staticExecOutputsInfo

  -- override Types.UnFixedInputType

  self.myUnFixComboType = Types.ClassifyType:extend();
  self.myUnFixComboType:SetData({},{});

  local pinInfo = staticDataInputsInfo[1]
  self.infoTable[BD.PIN_DATA_INPUT][1] =  staticDataInputsInfo[1]
  pinInfo = staticDataInputsInfo[2]
  self.infoTable[BD.PIN_DATA_INPUT][2] = BluePinInfo(BD.PIN_DATA_INPUT, pinInfo.argId, self.myUnFixComboType, pinInfo.name,pinInfo.tips,pinInfo.default,pinInfo.delAble, pinInfo.groupName);
  pinInfo = staticDataInputsInfo[3]
  self.infoTable[BD.PIN_DATA_INPUT][3] =  staticDataInputsInfo[3]
  pinInfo = staticDataInputsInfo[4]
  self.infoTable[BD.PIN_DATA_INPUT][4] =  staticDataInputsInfo[4]
  pinInfo = staticDataInputsInfo[5]
  self.infoTable[BD.PIN_DATA_INPUT][5] =  staticDataInputsInfo[5]
  pinInfo = staticDataInputsInfo[6]
  self.infoTable[BD.PIN_DATA_INPUT][6] =  staticDataInputsInfo[6]
end

function HandIndex:_GetCompCurrentDetectType(comp)
  local configs = {};
  local hints = {};
  if comp.FaceExpression then
    for k,v in pairs(cv.ClassifyComponent:GetFaceTypes()) do
      table.insert(configs, k)
      table.insert(hints, v)
    end
  end
  if comp.HandClassify then
    for k,v in pairs(cv.ClassifyComponent:GetHandTypes()) do
      table.insert(configs, k)
      table.insert(hints, v)
    end
  end
  return configs, hints ;
end

function HandIndex:_IsCompPin(pin)
  return pin.pinType == BD.PIN_DATA_INPUT and pin.argId == 1
end


---- 编辑器模式

if _KRATOSEDITOR then

  function HandIndex:_OnLink(selfPin, otherNode, otherPin)
    if self:_IsCompPin(selfPin) then
      -- 上一个节点必须是组件引用节点
      if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp ~= nil then
        if venuscore.isNil(otherNode.comp) then
          ERROR(ERROR_STR2)
        else
          local configs, hints = self:_GetCompCurrentDetectType(otherNode.comp);
          self.myUnFixComboType:SetData(configs, hints);
          self.inputs[2].literal = 0
        end
      else
        ERROR(ERROR_STR)
        ERROR("[HandIndex] error "..tostring(otherNode:GetFunctionType())..","..tostring(otherNode.comp))
      end
    end
  end

  function HandIndex:_OnUnLink(unlinkPin)
    if self:_IsCompPin(unlinkPin) then -- 断开是数据输入第一个引脚
      self:_Reset();
    end
  end


  function HandIndex:_OnCreateWithEditor()
    HandIndex.super._OnCreateWithEditor(self);
    self:_SetupInstanceInfoTable();
  end

  function HandIndex:_OnUpdateByEditor()

    HandIndex.super._OnUpdateByEditor(self);

    -- 如果编辑器显示当前蓝图,即时显示检查结果
    self:_OnCompile();
  end

end

HandIndex:RegisterInput(1, cv.ClassifyComponent:RTTI(), "component", "监听组件", BlueSelf());
HandIndex:RegisterInput(2, Types.ClassifyType, "eventType", "监听事件类型", 0);
HandIndex:RegisterInput(3, Types.BoolType, "Edge", "只在事件触发时执行(非持续)" , true)
HandIndex:RegisterInput(4, Types.FloatType, "imageRatio", "图片宽高比, w/h" , 1.0)
HandIndex:RegisterInput(5, Types.FloatType, "distance", "素材距离" , 0.16)
HandIndex:RegisterInput(6, Types.FloatType, "renderScale", "渲染比例" , 3.0)

HandIndex:RegisterOutput(1, Types.IntType , "HumanID", "人物序号");
HandIndex:RegisterOutput(2, mf.vector2:RTTI() , "Position", "框位置");
HandIndex:RegisterOutput(3, mf.vector2:RTTI() , "Size", "框大小");
HandIndex:RegisterOutput(4, mf.vector2:RTTI() , "Point", "回归点");
HandIndex:RegisterOutput(5, Types.BoolType , "isHigh", "是否高电平");
HandIndex:RegisterOutput(6, Types.BoolType , "isEdge", "是否边沿");
HandIndex:RegisterOutput(7, Types.BoolType , "isFingerUp", "是否食指向上触发");
HandIndex:RegisterOutput(8, Types.FloatType , "Scale", "渲染比例");


HandIndex:RegisterExecOutput(1, "OnTrigger", "执行");
HandIndex:RegisterExecOutput(2, "OnRelease", "释放");
--HandIndex:RegisterExecOutput(2, "DelayRelease", "放");

HandIndex:SetFunctionName("HandIndex");

HandIndex:SetFunctionType(BD.EVENT_FUNCTION_NODE);

HandIndex:SetEventType(BD.EventType.RegisterCallback);

return HandIndex;

