local venuscore = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venusjson = require "venusjson"
local apollodefine = require "apolloengine.defined"
local DeviceResource = require "apolloengine.deviceresource"
local ED = require "bluecore.eventdispatcher" 
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local json = require "cjson"
  
local TC = require "bluecore.base.transaction_center"

local VenusCache = apolloengine.VenusCache
local PhotoUpload = venuscore.VenusBehavior:extend("PhotoUpload");

--FIXME 菜单OpenScene的时候重定向纹理会错误 因为Game的根节点没有删除 等dev_prefab分支发版后可修复此问题
PhotoUpload:MemberRegister("TextureType",
  venuscore.ScriptTypes.ComboType(
    {
      {
        key = "photo",
        value = BD.RequestType.PHOTO
      },
      {
        key = "video",
        value = BD.RequestType.VIDEO
      },
      {
        key = "photovideo", 
        value = BD.RequestType.PHOTO_VIDEO  
      },
    }
  ));
PhotoUpload:MemberRegister("SizeLimit");
PhotoUpload:MemberRegister("SizeRange");
PhotoUpload:MemberRegister("AspectLimit");
PhotoUpload:MemberRegister("AspectRange");
PhotoUpload:MemberRegister("FaceLimit");
PhotoUpload:MemberRegister("TextureNum");
PhotoUpload:MemberRegister("DefaultTexturePathSet");
PhotoUpload:MemberRegister("RecommendTexturePathSet");

function PhotoUpload:new()
  self.TextureType = 0.0;
  self.SizeLimit = true;
  --这里让客户端针对大图进行降采样到1080P以下再发送对应路径给引擎
  self.SizeRange = mathfunction.vector4(360, 99999, 480, 99999); --宽 高限制
  self.AspectLimit = true;
  self.AspectRange = mathfunction.vector2(1/5, 5); --长宽比限制
  self.FaceLimit = false; --上传图片是否需要人脸
  self.TextureNum = 1; --
  self.DefaultTexturePathSet = {"comm:documents/texture/material/blank.jpg"};
  self.RecommendTexturePathSet = {{}};
  
  self.stickerPath = nil;
  self.cachePath = nil; --缓存图片
  self.transactionId = nil;
end


function PhotoUpload:CreateTexEntity(path)
  local tex = apolloengine.TextureEntity();
  tex:PushMetadata(apolloengine.TextureFileMetadata(
      apolloengine.TextureEntity.TU_STATIC,
      apolloengine.TextureEntity.PF_AUTO, 1, false,
      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
      apolloengine.TextureEntity.TF_LINEAR,
      apolloengine.TextureEntity.TF_LINEAR,
      path, false));
  tex:SetKeepSource(true);
  tex:CreateResource();
  return tex;
end

--[[
{
    "rec_source": ["src1", "src2"],
    "restrictions": [{
        "type": "size",
        "restrict": [{
            "width_min": 100,
            "width_max": 1000
        }, {
            "height_min": 100,
            "height_max": 1000
        }]
    }, {
        "type": "ratio",
        "restrict": [{
            "min": 0.2,
            "max": 5
        }]
    }, {
        "type": "face",
        "restrict": {
            "exists": "true"
        }
    }]
}
]]--

--替换proj:
function PhotoUpload:_ChangePathSet(path)
  local strTable = string.split(path, ':');
  return self.stickerPath..strTable[2];
end

function PhotoUpload:RequestData()
  local requestjson = {
    rec_source = {}, 
    restrictions = {}
};
  if next(self.RecommendTexturePathSet[1]) then
    local resourceTable = {}; 
    for _,path in pairs(self.RecommendTexturePathSet[1]) do
      table.insert(resourceTable, self:_ChangePathSet(path));
    end
    --local resourceTable = {self:_ChangePathSet(self.RecommendTexturePathSet[1][1])};
    requestjson["rec_source"] = resourceTable;
  end
  if self.SizeLimit then
    local sizeTable = {
                        type = "size",
                        restrict = {
                                      {
                                        width_min = 0,
                                        width_max = 0,
                                        height_min = 0,
                                        height_max = 0
                                      }
                                    }
                      };
    sizeTable.restrict[1].width_min = self.SizeRange:x();
    sizeTable.restrict[1].width_max = self.SizeRange:y();
    sizeTable.restrict[1].height_min = self.SizeRange:z();
    sizeTable.restrict[1].height_max = self.SizeRange:w();
    table.insert(requestjson["restrictions"], sizeTable);
  end
  if self.AspectLimit then
    local aspectTable = {
                          type = "ratio",
                          restrict = {{
                              min = 0,
                              max = 0
                          }}
                        };
    aspectTable.restrict[1].min = self.AspectRange:x();
    aspectTable.restrict[1].max = self.AspectRange:y();
    table.insert(requestjson["restrictions"], aspectTable);
  end
  if self.FaceLimit then
    local faceTable = {
                        type = "face",
                        restrict = {
                            exists = true
                        }
                      };
    table.insert(requestjson["restrictions"], faceTable);
  end

  local jsonstr = json.encode(requestjson);
  ERROR("PhotoUpload "..jsonstr); 
  return jsonstr
end

--回调ScreenImage HeadTrack改变shader宽高比参数
function PhotoUpload:RecallImageChange(node, tex, redirectName)
  local scriptCom = node:GetComponent(apolloengine.Node.CT_SCRIPT);
  if scriptCom and scriptCom.Instances[apollodefine.screenImageBehavior] then
    local ins = scriptCom.Instances[apollodefine.screenImageBehavior];
    ins:PhotoUploadRecall(tex, redirectName);
  end
  if scriptCom and scriptCom.Instances[apollodefine.screenImageBGBehavior] then
    local ins = scriptCom.Instances[apollodefine.screenImageBGBehavior];
    ins:PhotoUploadRecall(tex, redirectName);
  end
  local renderCom = node:GetComponent(apolloengine.Node.CT_RENDER);
  if renderCom and renderCom.MaterialEntities[1] then
    local shaderPath = renderCom.MaterialEntities[1]:GetShaderPath();
    shaderPath = string.gsub(shaderPath, "//", "/"); --避免路径多一/匹配不正确
    if shaderPath == apollodefine.htStickerShader then
      local slotNameList, slot = renderCom:GetUniformParameter(1); --headtrackNode只有一个matentity
      if slot["_MainTex"]:GetName() == redirectName then
        local texSize = tex:GetSize();
        --按理说设置headtrack相关的推荐素材的宽高比要卡成定值  要不然不好reset回去
        renderCom:SetParameter("_TexAspect", mathfunction.vector1(texSize.mx/texSize.my)); --设置给shader...
      end
    end
  end
  for _,subNode in pairs(node:GetChildrens()) do
    self:RecallImageChange(subNode, tex, redirectName);
  end
  -- body
end

--发送请求的object需要重写以下三个函数
function PhotoUpload:_IsTimeOut(deltaMs)
end

function PhotoUpload:_TimeOut()
end

function PhotoUpload:_Ready(jsonData)
  local jsonTbl = venusjson.LoadJsonString(jsonData)
  local path = jsonTbl["path"];
  --ERROR("_Ready is "..jsonData);
  if path and venuscore.IFileSystem:isFileExist(path) then
    self.cachePath = path;
    local tex = self:CreateTexEntity(path);
    DeviceResource:PushDeviceResource(DeviceResource.DEVICE_PHOTOUPLOAD1, tex);
    self:RecallImageChange(self.Node, tex, DeviceResource.DEVICE_PHOTOUPLOAD1);
    if ED:HasCompListener(self._hostID) and ED:HasEventListener(self._hostID, BD.Event.PhotoUploadResponse) then
      local event = {
          staticID = self._hostID;
          eventType = BD.Event.PhotoUploadResponse;
          params = {};
      };
      ED:PushEvent(event);
    end
  end
  --collectgarbage();
  return true -- persist request
end

function PhotoUpload:_CreateTransactionId()
  local count = 1.0;
  for i = 1, 10 do
    local tId = math.ceil(math.random() * 1000000);
    if not TC:HasTransaction(tId) then
      return tId;
    end
  end
  return -1;
end


function PhotoUpload:_OnStart()
end

--先记录下来 防止加载多场景的时候proj:绝对路径变化？
function PhotoUpload:_OnAwake() 
  self.stickerPath = venuscore.IFileSystem:PathAssembly("proj:");
  local cachePathSet = VenusCache:GetStickerTexCache(self.stickerPath);
  if next(cachePathSet) ~= nil then
    self.cachePath = cachePathSet[1];
    VenusCache:ClearStickerTexCache(self.stickerPath); --用了以后清掉？
    local tex = self:CreateTexEntity(self.cachePath);
    DeviceResource:PushDeviceResource(DeviceResource.DEVICE_PHOTOUPLOAD1, tex);
    self:RecallImageChange(self.Node, tex, DeviceResource.DEVICE_PHOTOUPLOAD1);
  else
    DeviceResource:PushDeviceResource(DeviceResource.DEVICE_PHOTOUPLOAD1, self:CreateTexEntity(self.DefaultTexturePathSet[1]));
  end
  if not _KRATOSEDITOR then --编辑器模式下不发送请求
    local tId = self:_CreateTransactionId();
    if tId ~= -1 then
      self.transactionId = tId;
      TC:Request(self, self.transactionId, self.TextureType, 0, self:RequestData());
    else
      ERROR("[PhotoUpload] Can Not Create Appropriate ID");
    end
  end

  --[[for idx, path in pairs(self.DefaultTexturePathSet) do
    local tex = apolloengine.TextureEntity();
    tex:PushMetadata(apolloengine.TextureFileMetadata(
        apolloengine.TextureEntity.TU_STATIC,
        apolloengine.TextureEntity.PF_AUTO, 1, false,
        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
        apolloengine.TextureEntity.TF_LINEAR,
        apolloengine.TextureEntity.TF_LINEAR,
        path));
    tex:CreateResource();
    DeviceResource:PushDeviceResource("PHOTO_UPLOAD"..tostring(idx), tex);  
  end]]--
end

function PhotoUpload:_OnUpdate(def)

end

function PhotoUpload:_OnDestroy()
  if self.transactionId then
    TC:Clear(self.transactionId)
    if self.cachePath then --fIXME 多张图片也需要修改
      VenusCache:PushStickerTexCache(self.stickerPath, self.cachePath);
    end
  end
end

function PhotoUpload:GetResourceDependencies()
  local res = {};
  for _,path in pairs(self.DefaultTexturePathSet) do
    table.insert(res, path);
    if BU:GetTextureMeta(path) then
      table.insert(res, path..".meta");
    end
  end
  for _,pathSet in pairs(self.RecommendTexturePathSet) do
    for _,path in pairs(pathSet) do
      table.insert(res, path);
      if BU:GetTextureMeta(path) then
        table.insert(res, path..".meta");
      end
    end
  end
  return res;

end


return PhotoUpload;