require "utility"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"

local shinelight = {}

function shinelight:Initialize(host, size)
  self.Queue = 100;
  self.BlurCount = 3;
  self.threshold = mathfunction.vector1(0.9);--高光亮度阈值
  self.rayColor = mathfunction.vector3(255.0, 255.0, 255.0);--光束颜色
  self.rayLength = 10;--光束长度
  self.rayLight = 0.5;--光束强度
  self.center = mathfunction.vector2(0.61, -0.55);--光束发射中心
  self.rayColorRatio = mathfunction.vector1(0.6);--光束颜色比例，0光束的颜色是选择的颜色，1就是高光区域原本颜色
  self.BlendMode  = mathfunction.vector1(2);

  self.POST_EFFECT_ALPHA = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "POST_EFFECT_ALPHA");

  self:_InitGreyPass(host, size);
  self:_InitBlurPass(host, size);
  self:_InitVolumnPass(host, size);
  self:_InitBlendPass(host, size);

  return self.Queue;
end

function shinelight:_InitGreyPass(host, size)
  self.RGB_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "RGB_TEXTURE");
  self.THRESHOLD = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "THRESHOLD");

  self.GreyRO = host:CreateRenderObject("comm:documents/shaders/posteffect/shinelight.material");
  self.GreyRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_A, size);

  self.GreyRO:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));

  host:RegisterScriptParameter(self, "threshold");
end

function shinelight:_InitBlurPass(host, size)	
  self.SHINE_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "SHINE_TEXTURE");   
  self.BLUR_RADIUS = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "BLUR_RADIUS");  
  self.UNIFORM_OFFSET = apolloengine.IMaterialSystem:NewParameterSlot(
  	apolloengine.ShaderEntity.UNIFORM,
    "UNIFORM_OFFSET");

  self.blurXRO = host:CreateRenderObject("comm:documents/shaders/posteffect/shinelightblur.material");  
  self.blurYRO = host:CreateRenderObject("comm:documents/shaders/posteffect/shinelightblur.material");
  self.blurXRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_B, size);
  self.blurYRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_C, size);

  self.blurXRO:SetParameter(self.BLUR_RADIUS, mathfunction.vector1(3.0));
  self.blurXRO:SetParameter(self.UNIFORM_OFFSET, mathfunction.vector2(1.0 / size:x(), 0.0));

  self.blurYRO:SetParameter(self.BLUR_RADIUS, mathfunction.vector1(3.0));
  self.blurYRO:SetParameter(self.UNIFORM_OFFSET, mathfunction.vector2(0.0, 1.0 / size:y()));

end

function shinelight:_InitVolumnPass(host, size)     
  self.BRIGHT_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "BRIGHT_TEXTURE");
  self.UNIFORM_LIGHT_COLOR = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "UNIFORM_LIGHT_COLOR");
  self.CENTER = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "CENTER");
  self.SAMPLES = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "SAMPLES"); 
  self.STEP = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "STEP"); 
  self.RATIO = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "RATIO");

  self.VolumnRO = host:CreateRenderObject( "comm:documents/shaders/posteffect/shinevolumnlight.material");
  self.VolumnRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_D, size );
  self.VolumnRO:SetParameter(self.SAMPLES, mathfunction.vector1(100));
  self.VolumnRO:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));

  host:RegisterScriptParameter(self, "rayColor");
  host:RegisterScriptParameter(self, "rayLength");
  host:RegisterScriptParameter(self, "center");
  host:RegisterScriptParameter(self, "rayLight");
  host:RegisterScriptParameter(self, "rayColorRatio");
end

function shinelight:_InitBlendPass(host, size)
  self._BlendMode = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendMode");
  self.BlendOpacity = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendOpacity");
  self._MainTex = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_MainTex");   
  self._BlendTex = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendTex");
  self.BlendRO = host:CreateRenderObject("comm:documents/shaders/posteffect/blendmode.material");
  self.BlendRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_E, size );
  self.BlendRO:SetParameter(self.BlendOpacity, mathfunction.vector1(2.0));
  self.BlendRO:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));

  host:RegisterScriptParameter(self, "BlendMode");
end

function shinelight:_DrawGreyPass(pipeline, Original, Scene, Output)
  self.GreyRT:PushRenderTarget();
  self.GreyRT:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR);

  self.GreyRO:SetParameter(self.RGB_TEXTURE, Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.GreyRO:SetParameter(self.THRESHOLD, self.threshold);

  self.GreyRO:Draw(pipeline);
end

function shinelight:_DrawBlurPass(pipeline, Original, Scene, Output)
  local last = self.GreyRT;
  for i = 1, self.BlurCount do
    self.blurXRT:PushRenderTarget();
    self.blurXRT:ClearBuffer( apolloengine.RenderTargetEntity.CF_COLOR );
    self.blurXRO:SetParameter(self.SHINE_TEXTURE, last:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0)); 
    self.blurXRO:Draw(pipeline);

    self.blurYRT:PushRenderTarget();
    self.blurYRT:ClearBuffer( apolloengine.RenderTargetEntity.CF_COLOR );
    self.blurYRO:SetParameter(self.SHINE_TEXTURE, self.blurXRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
    self.blurYRO:Draw(pipeline);

    last = self.blurYRT;
  end
end

function shinelight:_DrawVolumnPass(pipeline, Original, Scene, Output)
  self.VolumnRT:PushRenderTarget();
  self.VolumnRT:ClearBuffer( apolloengine.RenderTargetEntity.CF_COLOR );

  self.VolumnRO:SetParameter(self.BRIGHT_TEXTURE, self.blurYRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  local color = (self.rayColor / 255) * self.rayLight;
  self.VolumnRO:SetParameter(self.UNIFORM_LIGHT_COLOR, color);
  self.VolumnRO:SetParameter(self.CENTER, self.center);
  local step = mathfunction.vector1(0.005) * self.rayLength;
  self.VolumnRO:SetParameter(self.STEP, step);
  local ratio = self.rayColorRatio;
  self.VolumnRO:SetParameter(self.RATIO, ratio);
  self.VolumnRO:Draw(pipeline);
end

function shinelight:_DrawBlendPass(pipeline, Original, Scene, Output)
  Output:PushRenderTarget();
  Output:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR);
  self.BlendRO:SetParameter(self._MainTex, Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.BlendRO:SetParameter(self._BlendTex, self.VolumnRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.BlendRO:SetParameter(self._BlendMode, self.BlendMode);

  self.BlendRO:Draw(pipeline);
end

function shinelight:Resizeview(size)  
  self.blurXRO:SetParameter( self.UNIFORM_OFFSET, mathfunction.vector2( 1.0 / size:x(), 0 ) );
  self.blurYRO:SetParameter( self.UNIFORM_OFFSET, mathfunction.vector2( 0, 1.0 / size:y() ) );
end

function shinelight:Process(pipeline, Original, Scene, Output)
  self:_DrawGreyPass(pipeline, Original, Scene, Output);
  self:_DrawBlurPass(pipeline, Original, Scene, Output);
  self:_DrawVolumnPass(pipeline, Original, Scene, Output);
  self:_DrawBlendPass(pipeline, Original, Scene, Output);
end

return shinelight;  

