local venuscore = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"

local TouchCacheBehavior = venuscore.VenusBehavior:extend("TouchCacheBehavior");

local TRANSFORM_CACHE_NAME = "transform_cache";
local TRANSFORM_PROPERTIES = {"POSTION.X", "POSTION.Y" --[[, "POSTION.Z", "ROTATION.X", "ROTATION.Y", "ROTATION.Z", "SCALE.X", "SCALE.Y", "SCALE.Z"]]};

function TouchCacheBehavior:new()
	self.transformComp = nil
	self.cameraComp = nil
	self.animationComp = nil
	self.renderBufferComp = nil
	self.particleCompList = {}
	self.cacheing = false
	self.currentTs = 0
  self.lastTouchPosition = nil
  self.tranformCacheName = nil
end

function TouchCacheBehavior:Initialize(cameraCom, animCacheIndex)
    self.cameraComp = cameraCom
    self.tranformCacheName = TRANSFORM_CACHE_NAME .. "_" .. tostring(animCacheIndex)

    self.transformComp = self.Node:GetComponent(apolloengine.Node.CT_TRANSFORM);
    if self.transformComp ~= nil then
        self.animationComp = self.Node:CreateComponent(apolloengine.Node.CT_ANIMATION);
    else
        LOG("transformComp not found")
    end

    self:_FindParticleComp(self.Node, self.particleCompList)
    local subNodes = self.Node:GetChildrens();
    for i = 1, #subNodes do
        self:_FindParticleComp(subNodes[i], self.particleCompList)
    end

    if not self:_IfParticleExist() then
        LOG("particleComponent not found")
        return;
    end

    self.renderBufferComp = self.Node:CreateComponent(apolloengine.Node.CT_RENDER_BUFFER);
    if not self.renderBufferComp then
      LOG("renderBufferComp create failed")
      return;
    end

    for index = 1, #self.particleCompList do
      local particleComp = self.particleCompList[index]
      if particleComp then
          particleComp:Play();
      end
    end
end

function TouchCacheBehavior:StartCache()
    LOG("TouchCacheBehavior:StartCache")
    if self.cacheing ~= true then
        self:_StartTransformCache()
        self:_StartParticleCache()
        self.cacheing = true;

        if self.lastTouchPosition ~= nil then
          LOG("TouchCacheBehavior:StartCache _SetPosition")
          self:_SetPosition(self.lastTouchPosition)
          self.lastTouchPosition = nil
        end
    end
end

function TouchCacheBehavior:StopCache(curVideoPlayTs)
    LOG("TouchCacheBehavior:StopCache ".. curVideoPlayTs)
    if self.cacheing then
        self.cacheing = false;
        self:_StopTransformCache(curVideoPlayTs)
        self:_StopParticleCache()
    end
end

function TouchCacheBehavior:Update(deltaTs)
  --LOG("TouchCacheBehavior:Update " .. tostring(deltaTs))
  if not self.cacheing then
    return
  end
  
  self.currentTs = self.currentTs + deltaTs
  self:_DoTransformCache(self.currentTs)
end

function TouchCacheBehavior:_FindParticleComp(node, particleCompList)
  local particleComp = node:GetComponent(apolloengine.Node.CT_PARTICLE);
  if particleComp ~= nil then
      table.insert(particleCompList, particleComp)
  end
end

function TouchCacheBehavior:_IfParticleExist()
  return #self.particleCompList > 0
end

function TouchCacheBehavior:SetStartTs(startTime)
  self.startTs = startTime
end

function TouchCacheBehavior:_StartTransformCache()
  self.currentTs = 0
  if self.animationComp ~= nil then
      self.animationComp:CreateMemoryKeyFrame(self.tranformCacheName);
      for proKey, proValue in pairs(TRANSFORM_PROPERTIES) do
          self.animationComp:CreateCurve(self.tranformCacheName, self.transformComp, proValue);
      end
  end
end

function TouchCacheBehavior:_StopTransformCache(curVideoPlayTs)
    if self.animationComp then
        self.animationComp:Play(self.tranformCacheName);
        self.animationComp:SetCurrentTimeToEnd();
        if self.startTs == nil then
          self.startTs = curVideoPlayTs
        end
        self.animationComp:SetAnimationWorkTime(self.tranformCacheName, self.startTs * 0.001, curVideoPlayTs * 0.001);
    end
end

function TouchCacheBehavior:_StartParticleCache()
  if self.renderBufferComp ~= nil then
      self.renderBufferComp:StartBuffer()
  end
end

function TouchCacheBehavior:_StopParticleCache()
    for index = 1, #self.particleCompList do
        LOG("to call StopEmitting")
        self.particleCompList[index]:StopEmitting()
    end
    if self.renderBufferComp ~= nil then
        self.renderBufferComp:StopBuffer()
    end
end

function TouchCacheBehavior:SetTouchPosition(posVec2)
  --LOG("[SetTouchPosition]input  " .. posVec2:x() .. ", " .. posVec2:y())
  self.lastTouchPosition = posVec2
  self:_SetPosition(posVec2)
end

function TouchCacheBehavior:_SetPosition(posVec2)
    if self.cacheing and self.transformComp and self.cameraComp then
        local curPos = self.transformComp:GetWorldPosition();
        local point = mathfunction.vector3(0.0, 0.0, curPos:z())
        local normal = mathfunction.vector3(0.0, 0.0, 1.0)
        local worldPlane = mathfunction.Plane(point, normal);

        local ray = self.cameraComp:PickRay(posVec2);
        local intersect_point = worldPlane:IntersectPoint(ray);

        self.transformComp:SetWorldPosition(intersect_point)

        --LOG("[_SetPosition]input  " .. posVec2:x() .. ", " .. posVec2:y())
        --LOG("[_SetPosition] intersect ".. intersect_point:x() .. ", ".. intersect_point:y() .. ", " .. intersect_point:z())
        
        self.lastTouchPosition = nil
    end
end

function TouchCacheBehavior:_DoTransformCache(currentTs)
    if self.animationComp ~= nil then
      for proKey, proValue in pairs(TRANSFORM_PROPERTIES) do
          self.animationComp:AddKeyFrameToCurve(self.transformComp, self.tranformCacheName, proValue, currentTs);
      end
    end
end

function TouchCacheBehavior:SetParticleScale(scale)
  LOG(string.format("SetParticleScale scale=%d", scale));
  for index = 1, #self.particleCompList do
      self.particleCompList[index]:SetAdjustSize(scale)
  end
end

--rgb color
function TouchCacheBehavior:SetParticleColor(color)
  LOG(string.format("SetParticleColor color=%d", color));
  for index = 1, #self.particleCompList do
      self.particleCompList[index]:SetAdjustColor(color)
  end
end

--- 一个素材可能有多个particleComponent
function TouchCacheBehavior:GetParticleScale()
  local index = #self.particleCompList
  return self.particleCompList[index]:GetAdjustSize()
end

--rgb color
function TouchCacheBehavior:GetParticleColor()
  local index = #self.particleCompList
  return self.particleCompList[index]:GetAdjustColor()
end

--预览粒子效果
function TouchCacheBehavior:StartPreviewTouchMagic()
  for index = 1, #self.particleCompList do
    -- 笔刷素材是距离模式，预览时没有效果。这里判断一下，设置按时间生成粒子
    local emitter = self.particleCompList[index]:GetEmitter();
    local rateOverTime = emitter.EmissionRateOverTime;
    if rateOverTime.ConstMinValue == 0 then
      rateOverTime.ConstMinValue = 10
    end
    
    self.particleCompList[index]:SetParticleVisibility(true)
  end
end

--结束预览粒子效果
function TouchCacheBehavior:StopPreviewTouchMagic()
  for index = 1, #self.particleCompList do
    self.particleCompList[index]:SetParticleVisibility(false)
    self.particleCompList[index]:Stop()

     --重置状态，使结点即可以预览也可以应用
    self.particleCompList[index]:Reset()
    self.particleCompList[index]:Play()
  end
end

function TouchCacheBehavior:SetParticleVisibility(show)
  for index = 1, #self.particleCompList do
    self.particleCompList[index]:SetParticleVisibility(show)
  end
end

return TouchCacheBehavior;
