local venuscore = require "venuscore"
local venusjson = require "venusjson"
local apolloengine = require "apollocore"
local apolloDefine = require "apolloutility.defiend"
local BundleSystem = require "venuscore.bundle.bundlesystem"
local mathfunction = require "mathfunction"
local likeapp = require "likeapp"
local utils = require "behavior.cartoon_behavior.utils"
local defined = require "behavior.cartoon_behavior.defined"

local Load = venuscore.VenusBehavior:extend("LoadBehavior")
local DEFORMER_SCRIPT = "scrs:behavior/cartoon_behavior/deformer_behavior.lua"
local FACE_TRANSFER_SCRIPT = "facetransfer_behavior.lua"
local ANIMATION_PLAY_SCRIPT = "animation_play_behavior"
local RESOURCE_KEY = {DEFAULT_KEY = "default", BALD_KEY = "bald"}

function Load:new()
  Load.super.new(self)
  self.isInited = false
  self.animation_start = false
  self.first_frame = true
  self.animation_behaviors = {}

  self.use_default = {
    gender = true,
    albedo_color = true,
    hair_color = true,
    models = defined.resource_status.DEFAULT,
    id_weights = true
  }
end

function Load:_OnAwake()
end

function Load:_OnUpdate(delta)
  if not self.isInited then
    self:Init()
    return
  end

  if self.first_frame then  -- the delta includes the init time
    self.first_frame = false
    return
  end

  if not self.animation_start then
    self:PlayAnimations()
    self.animation_start = true
  end
  
  self:CheckAnimationFinished()
end

function Load:RegisterAnimationNode(node)
  local behavior = utils:GetBehavior(node, ANIMATION_PLAY_SCRIPT)
  table.insert(self.animation_behaviors, behavior)
end

function Load:Init()
  self:LoadResources()  
  self.isInited = true
  --os.execute("ping -n " .. tonumber(10) .. " localhost > NUL")
end

function Load:LoadResources()
  LOG("CARTOON_TIME: start to get res")
  local res = self:GetFaceConfig()
  LOG("CARTOON_TIME: finish to get res")
  if res == "" or res == nil then
    WARNING("CARTOON: res is empty, use default resource")
    return
  end
    
  local male = self:IsMale(res)
  local default_nodes = self:GetNodesToChange(male)
  LOG("CARTOON_TIME: load prefabs")
  
  local custom_nodes = self:LoadCustomModels(res, default_nodes.face)
  self:InactiveDefaultNodes(default_nodes, custom_nodes)

  LOG("CARTOON_TIME: change skin color")
  self:ChangeSkinColor(res, default_nodes)
  
  LOG("CARTOON_TIME: change id weights")
  self:ChangeIdWeight(res, default_nodes.face)  
  
  LOG("CARTOON_TIME: change hair color")
  self:ChangeHairColor(res, default_nodes, custom_nodes)
  
  LOG("CARTOON_TIME: finish loading")
end

function Load:PlayAnimations()
  for _, behavior in ipairs(self.animation_behaviors) do
    behavior:Play()
  end
end

function Load:GetFaceConfig()
  if _PLATFORM_WINDOWS then
    local path = "E:/data/cartoon/resources/res.json"
    if not venuscore.IFileSystem:isFileExist(path) then
      return nil
    end
    return venusjson.LaodJsonFile(path)
  end
  if likeapp.AI.GetCustomJsonString == nil then    
    WARNING("CARTOON: no GetCustomJsonString interface")
    return nil
  end
  local res = likeapp.AI:GetCustomJsonString()
  if res == "" then
    return res
  end

  res = res:gsub("\\/", "/")
  res = venusjson.LoadJsonString(res)
  return res
end

function Load:CheckAnimationFinished()
  local playing = false
  local max_code = 0
  for _, behavior in ipairs(self.animation_behaviors) do
    local status = behavior:GetStatus()
    if status == defined.animation_status.PLAYING then
      playing = true
    end
    if status > max_code then
      max_code = status
    end
  end
  
  if not playing then
    local animation_result = self:GetResultFromAnimation(max_code)
    local resource_result = self:GetResourceStatus()
    local result = defined.result_code[animation_result][resource_result]
    WARNING("CARTOON: animation finished with code "..result)
    
    if likeapp.AI.GetCustomAnimationState ~= nil then
      likeapp.AI:GetCustomAnimationState(result)
    end
  end
end

function Load:GetResultFromAnimation(status)
  if status == defined.animation_status.DONE then
    return "SUCCESS"
  elseif status == defined.animation_status.STOP then
    return "STOP"
  end
  return "ERROR"
end

function Load:GetResourceStatus()
  if self.use_default.gender then
    LOG("CARTOON: use default gender")
  end
  if self.use_default.albedo_color then
    LOG("CARTOON: use default albedo_color")
  end
  if self.use_default.hair_color then
    LOG("CARTOON: use default hair_color")
  end
  if self.use_default.id_weights then
    LOG("CARTOON: use default id_weights")
  end
  LOG("CARTOON: use models status "..self.use_default.models)
  return self.use_default.models
end

function Load:LoadPrefabModels(config_models, face_node)    
  local face_render = face_node:GetComponent(apolloengine.Node.CT_RENDER)
  local prefab_nodes = {}
  for key, value in pairs(config_models) do
    local nodes = self:LoadPrefab(value, face_node)
    if nodes then
      self:BindTargetRender(nodes, face_render)
      prefab_nodes[key] = nodes
    end
  end
  return prefab_nodes
end

function Load:LoadPrefab(path, root)
  if path == nil or path == RESOURCE_KEY.DEFAULT_KEY then
    WARNING("CARTOON: prefab path is nil or default, use default model")
    return nil
  end
  
  local rootconfig = venusjson.LaodJsonFile(path)
  if not rootconfig then
    WARNING("CARTOON: failed load prefab json file, use default model: "..path)
    return nil
  end
  if not rootconfig.scene then
    WARNING("CARTOON: prefab format incorrect, missing scene, use default model: "..path)
    return nil
  end
  
  self:ChangeResource(self:GetDir(path).."/")
  local nodes = self:PrefabScene(rootconfig.scene)  
  self:ChangeResource(nil)
  
  if nodes then
    for _, node in ipairs(nodes) do
      root:AttachNode(node)
    end
  end
  
  return nodes
end

function Load:PrefabScene(scene_path)
  local bundlepath = venuscore.IFileSystem:PathAssembly(scene_path) 
  if not venuscore.IFileSystem:isFileExist(bundlepath) then
    WARNING("CARTOON: prefab scene not exist, use default model: "..bundlepath)
    return nil
  end
  
  --local str = self:ReadFile(bundlepath)
  local scene = apolloengine.SceneManager:GetOrCreateScene(apolloDefine.default_scene_name);
  local _,nodes = BundleSystem:DeserializeFromPath(bundlepath,BundleSystem.DeserializeMode.Prefab,scene);
  --local nodes = BundleSystem:DeSerialize(str,scene,BundleSystem.DeserializeMode.Prefab)
  if nodes == nil then
    WARNING("CARTOON: preface scene deserialize failed, use default model: "..bundlepath);
    return nil
  end
  
  if not _KRATOSEDITOR and self.camera then
    for _, node in ipairs(nodes) do    
      node:SetLayer(self.camera.LayerMask)
      local resType = self.camera:GetResourceType()
      local render = node:GetComponent(apolloengine.Node.CT_RENDER)
      if render ~= nil then
        render:SetResourceType(resType)
      end
    end
  end

  return nodes
end

function Load:ChangeResource(path)
  if path == nil then
    if self.origin_resource ~= nil then
      self:SetResource(self.origin_resource)
    end
  else
    self.origin_resource = self:SetResource(path)
  end
end

function Load:SetResource(path)
  if _KRATOSEDITOR then
    local origin = venuscore.IFileSystem:PathAssembly("proj:")
    venuscore.IFileSystem:SetProjectPath(path)
    return origin
  end
  local origin = venuscore.IFileSystem:GetResourcePath()
  venuscore.IFileSystem:SetResourcePath(path)
  return origin
end

function Load:IsMale(res)
  if res.male == nil then
    return false
  end
  self.use_default.gender = false
  return res.male == 1
end

function Load:LoadCustomModels(res, face_node)
  if face_node == nil
  or res.config == nil
  or res.config.models == nil
  then
    return {}
  end

  return self:LoadPrefabModels(res.config.models, face_node)
end

function Load:InactiveDefaultNodes(default_nodes, custom_nodes)
  local disable_cnt = 0
  if custom_nodes.hair then
    if default_nodes.hair then
      default_nodes.hair.Active = false
    end
    disable_cnt = disable_cnt + 1
  end
  if custom_nodes.eyebrow then
    if default_nodes.eyebrow_l then
      default_nodes.eyebrow_l.Active = false
    end
    if default_nodes.eyebrow_r then
      default_nodes.eyebrow_r.Active = false
    end
    disable_cnt = disable_cnt + 1
  end
  if disable_cnt == 1 then
    self.use_default.models = defined.resource_status.PARTIAL
  elseif disable_cnt == 2 then
    self.use_default.models = defined.resource_status.CUSTOM
  end
end

function Load:ChangeSkinColor(res, default_nodes)
  if res.albedo_color == nil then
    return
  end

  self.use_default.albedo_color = false
  local color_id = defined.server_color_type_id.albedo_color
  local value = self:GetColorValue(res.albedo_color)
  self:ChangeNodeColor(default_nodes.face, color_id, value)
  self:ChangeNodeColor(default_nodes.body, color_id, value)
end

function Load:ChangeIdWeight(res, face_node)
  if face_node == nil or res.config == nil or res.config.id_weights == nil then
    return
  end
  
  self.use_default.id_weights = false
  local transfer_behavior = utils:GetBehavior(face_node, FACE_TRANSFER_SCRIPT)
  if transfer_behavior then
    transfer_behavior:TransferWeight(res.config.id_weights)
  end
end

function Load:ChangeHairColor(res, default_nodes, custom_nodes)
  if res.hair_color == nil then
    return
  end
  
  self.use_default.hair_color = false
  if res.hair_color ~= nil then
    local color_id = defined.server_color_type_id.hair_color
    local value = self:GetColorValue(res.hair_color)
    self:ChangeNodeColor(default_nodes.eyelash, color_id, value)
    if custom_nodes then
      self:ChangeNodesColor(custom_nodes.hair, color_id, value)
      self:ChangeNodesColor(custom_nodes.eyebrow, color_id, value)
      self:ChangeNodesColor(custom_nodes.beard, color_id, value)
    end
  end
end

function Load:GetNodesToChange(male)
  if male then
    return {
      face = self.MaleFace,
      body = self.MaleBody,
      eyelash = self.MaleEyelash,
      hair = self.MaleHair,
      eyebrow_l = self.MaleLeftEyebrow,
      eyebrow_r = self.MaleRightEyebrow
    }
  end
  
  return {
      face = self.FemaleFace,
      body = self.FemaleBody,
      eyelash = self.FemaleEyelash,
      hair = self.FemaleHair,
      eyebrow_l = self.FemaleLeftEyebrow,
      eyebrow_r = self.FemaleRightEyebrow
  }
end

function Load:BindTargetRender(nodes, render)
  for _, node in ipairs(nodes) do
    local behavior = utils:GetBehavior(node, DEFORMER_SCRIPT)
    if behavior ~= nil then
      behavior:BindTargetRender(render)
    end
  end
end

function Load:ChangeNodesColor(nodes, type_id, value)
  if nodes == nil then
    return
  end
  for _, node in ipairs(nodes) do
    self:ChangeNodeColor(node, type_id, value)
  end
end

function Load:ChangeNodeColor(node, type_id, value)
  if node == nil then
    return
  end
  local render = node:GetComponent(apolloengine.Node.CT_RENDER)
  if render == nil then
    WARNING("no render component found")
    return
  end
  render:SetParameter(
      defined.color_uniform[type_id],
      value)
end

function Load:GetDir(path)
  return string.match(path, "(.+)/[^/]*%.%w+$")
end

function Load:ReadFile(path)
  local file = io.open(path, "rb")
  local str = file:read("*a")
  io.close(file)
  return str
end

function Load:WriteFile(path, str)
  local file = io.open(path, "w")
  file:write(str)
  io.close(file)
end

function Load:GetColorValue(color)
  return mathfunction.vector4(color[3] / 255.0, color[2] / 255.0, color[1] / 255.0, 1.0)
end

Load:MemberRegister(
  "camera",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.CameraComponent:RTTI(),    
    nil,
    function(thiz, value)
        thiz.camera = value
    end
  )
)

Load:MemberRegister(
  "FemaleFace",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleFace = value
    end
  )
)

Load:MemberRegister(
  "FemaleBody",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleBody = value
    end
  )
)

Load:MemberRegister(
  "FemaleHair",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleHair = value
    end
  )
)

Load:MemberRegister(
  "FemaleEyelash",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleEyelash = value
    end
  )
)

Load:MemberRegister(
  "FemaleLeftEyebrow",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleLeftEyebrow = value
    end
  )
)

Load:MemberRegister(
  "FemaleRightEyebrow",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.FemaleRightEyebrow = value
    end
  )
)

Load:MemberRegister(
  "MaleFace",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleFace = value
    end
  )
)

Load:MemberRegister(
  "MaleBody",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleBody = value
    end
  )
)

Load:MemberRegister(
  "MaleHair",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleHair = value
    end
  )
)

Load:MemberRegister(
  "MaleEyelash",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleEyelash = value
    end
  )
)

Load:MemberRegister(
  "MaleLeftEyebrow",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleLeftEyebrow = value
    end
  )
)

Load:MemberRegister(
  "MaleRightEyebrow",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
        thiz.MaleRightEyebrow = value
    end
  )
)

return Load