local BundleSystem = require "venuscore.bundle.bundlesystem"
local apolloengine = require "apollocore"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"
local apollocore = require "apollocore"
local venusjson = require "venusjson"
local likeapp = require "likeapp"

local EditorSDFSubtitleManager = {}


function EditorSDFSubtitleManager:Initialize(scene, sequenceBase)
    self.resourceType = sequenceBase
    self.sequenceBase = sequenceBase
    self.scene = scene;
    self.id = 0;
    self.rootNode = nil;
    self.curNodesMap = {};
    self.nodesMapBeforeEdit = {};
    self.inEditState = false;
    self.cameraRenderTarget = nil;
    --进入字幕页再加载emoji与字体相关信息
    LOG("[EditorSDFSubtitleManager] SDFGenerator Initialize");
    apollocore.ISDFTextSystem:Initialize();
end

function EditorSDFSubtitleManager:Exit()
  LOG("[EditorSDFSubtitleManager] Clear Font Resource");
  apollocore.ISDFTextSystem:ClearFontResource();
end

function EditorSDFSubtitleManager:SetMainrt(rt)
    LOG("[SetMainrt]")
    for index = 1, #self.curNodesMap do
        local node = self.curNodesMap[index]
        self:_ConfigRenderTarget(node, rt)
    end
    self.cameraRenderTarget = rt;
end

function EditorSDFSubtitleManager:SetDefaultFontPaths(fontPaths)
    LOG("[SetDefaultFonts]")
    apollocore.ISDFTextSystem:SetOSFonts(fontPaths)
end

function EditorSDFSubtitleManager:SetPadding(padding)
    LOG("[SetPadding]")
    apollocore.ISDFTextSystem.Padding = padding
    return true;
end

function EditorSDFSubtitleManager:EnterEditState()
    ERROR("[EnterEditState]")
    if self.inEditState then
        ERROR("[EnterEditState] in edit state");
        return false;
    end
    self.inEditState = true;
    if #self.nodesMapBeforeEdit > 0 then
        ERROR("[EnterEditState] ERROR!!! nodesMapBeforeEdit'size = "..tostring(#self.nodesMapBeforeEdit))
        self.nodesMapBeforeEdit = {};
    end
    for subtitleId, node in pairs(self.curNodesMap) do
        local instance = self:_GetNodeBehaviorInstance(subtitleId)
        if instance then
            instance:StoreAttributes()
        end
        self.nodesMapBeforeEdit[subtitleId] = node
    end
    ERROR("[EnterEditState]exit")
    return true;
end

function EditorSDFSubtitleManager:ExitEditState(save)
    ERROR("[ExitEditState] save = " .. tostring(save))
    if self.inEditState ~= true then
        ERROR("[ExitEditState] not in edit state");
        return false;
    end
    self.inEditState = false;

    if save then
        -- 需要将nodesMapBeforeEdit中保存的node和curNodesMap不一样的，给释放掉
        for subtitleId, legacyNode in pairs(self.nodesMapBeforeEdit) do
            local curNode = self.curNodesMap[subtitleId]
            if legacyNode ~= nil and  curNode ~= legacyNode then
                self:_ReleaseNode(legacyNode)
            end
        end
    else
        --先去掉新加的
        for subtitleId, curNode in pairs(self.curNodesMap) do
            local legacyNode = self.nodesMapBeforeEdit[subtitleId]
            if legacyNode == nil or legacyNode ~= curNode then
                LOG(tostring(subtitleId) .. " is new added")
                self:_ReleaseNode(curNode)
            end
        end
        self.curNodesMap = self.nodesMapBeforeEdit
        --重新恢复字幕属性并且激活
        for subtitleId, node in pairs(self.curNodesMap) do
            local instance = self:_GetNodeBehaviorInstance(subtitleId)
            if instance then
              local storeInfo = instance:GetStoreAttributes();
              if storeInfo then
                if storeInfo["color"] then
                  LOG("Store Color is "  .. storeInfo["color"]:x() .. "_" .. storeInfo["color"]:y() .. "_" .. storeInfo["color"]:z() .. "_" .. storeInfo["color"]:w())
                  instance:RestoreAttributes();
                  node.Active = true;
                else
                  ERROR("can not revert to Origin Color, so reload Resource again");
                  self:_ReleaseNode(node);
                  self.curNodesMap[subtitleId] = nil;
                  local newNode = self:_CreateNode(storeInfo["resourcePath"])
                  self.curNodesMap[subtitleId] = newNode

                  self:UpdateSubtitleFontPath(subtitleId, storeInfo["fontPath"])
                  self:UpdateSubtitleText(subtitleId, storeInfo["text"])
                  self:SetSubtitleFontPosition(subtitleId, storeInfo["pos"])
                  self:SetSubtitleFontRotation(subtitleId, storeInfo["rotation"])
                  self:SetSubtitleFontScale(subtitleId, storeInfo["scale"])
                  self:UpdateSubtitleTimeRange(subtitleId, storeInfo["timeRange"]:x(), storeInfo["timeRange"]:y())
                  self:UpdateFontColor(subtitleId, nil, storeInfo["useGradient"])
                  newNode.Active = true
                end
              end
            end
        end
    end
    self.nodesMapBeforeEdit = {};
    --清除缓存信息
    likeapp.SDFInfo:ClearSDFInfos()
    likeapp.TextureStreamRepo:Clear()
    likeapp.FontIndex:Clear()

    return true;
end

function EditorSDFSubtitleManager:AddSubtitle(resourcePath, textStr)
    ERROR("[AddSubtitle]" .. resourcePath)
    local node = self:_CreateNode(resourcePath)
    if node then
        node.Active = true
        local subtitleId = self:_GetSubtitleLogicId()
        self.curNodesMap[subtitleId] = node;
        LOG("[AddSubtitle] subtitleId = " .. tostring(subtitleId))
        if textStr then
            self:UpdateSubtitleText(subtitleId, textStr)
            local index = apollocore.ISDFTextSystem:GetFontIndex(textStr)
            if index ~= nil then
                likeapp.FontIndex:Set(textStr, index)
            end
        end
        return subtitleId;
    else
        return 0
    end
end

function EditorSDFSubtitleManager:RemoveSubtitle(subtitleId)
    ERROR("[RemoveSubtitle] subtitleId = " .. tostring(subtitleId))
    local node = self.curNodesMap[subtitleId]
    if node ~= nil then
        if self.nodesMapBeforeEdit[subtitleId] ~= nil then
            node.Active = false
        else
            self:_ReleaseNode(node)
        end
        self.curNodesMap[subtitleId] = nil
    end
    return true
end

function EditorSDFSubtitleManager:RemoveAllSubtitles()
    ERROR("[RemoveAllSubtitles]")

    for id, node in pairs(self.curNodesMap) do
        self:_ReleaseNode(node)
    end

    for id, node in pairs(self.nodesMapBeforeEdit) do
        self:_ReleaseNode(node)
    end

    self.curNodesMap = {};
    self.nodesMapBeforeEdit = {};

    likeapp.SDFInfo:ClearSDFInfos()
    likeapp.TextureStreamRepo:Clear()
    likeapp.FontIndex:Clear()

    return true;
end

function EditorSDFSubtitleManager:SetSubtitleVisible(subtitleId, visible)
    LOG("[SetSubtitleVisible] subtitleId = " .. tostring(subtitleId) .. ", visible = " .. tostring(visible))
    if self:_IsSubtitleExist(self.curNodesMap, subtitleId) ~= true then
        ERROR("[SetSubtitleVisible] not exist id " .. tostring(subtitleId))
        return false
    end
    self.curNodesMap[subtitleId].Active = visible
    return true
end

--注意：更新素材需要重新设置所有状态
function EditorSDFSubtitleManager:UpdateSubtitleResource(subtitleId, resourcePath)
    ERROR("[UpdateSubtitleResource] " .. resourcePath)
    if self:_IsSubtitleExist(self.curNodesMap, subtitleId) ~= true then
        ERROR("[UpdateSubtitleResource] not exist id " .. tostring(subtitleId))
        return false
    end

    --1.如果节点是进入编辑之前已存在的，为保证可以在退出编辑时正常恢复，不能将其移除，而是存放到待删除列表；
    --相反，如果在进入编辑之前不存在，那直接删除节点即可
    local legacyNode = self.curNodesMap[subtitleId]

    local info = self:_GetSubtitleInfo(subtitleId)

    if self.nodesMapBeforeEdit[subtitleId] == legacyNode then
        legacyNode.Active = false
    else
        self:_ReleaseNode(legacyNode)
    end

    --2.新建素材节点
    local newNode = self:_CreateNode(resourcePath)
    self.curNodesMap[subtitleId] = newNode

    --3.恢复原先模板信息
    if info then
        local transform = info["transform"]
        local timeRange = info["timeRange"]
        self:UpdateSubtitleText(subtitleId, info["text"])
        self:SetSubtitleFontPosition(subtitleId, mathfunction.vector2(transform:x(), transform:y()))
        self:SetSubtitleFontScale(subtitleId, transform:z(), true)
        self:SetSubtitleFontRotation(subtitleId, transform:w())
        self:UpdateSubtitleTimeRange(subtitleId, timeRange:x(), timeRange:y())
    end
    return true
end

function EditorSDFSubtitleManager:UpdateSubtitleText(subtitleId, textStr)
    ERROR("[UpdateSubtitleText]" .. tostring(textStr))
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:UpdateSubtitleText(textStr)
        local index = apollocore.ISDFTextSystem:GetFontIndex(textStr)
        if index ~= nil then
            likeapp.FontIndex:Set(textStr, index)
        end
        return true
    end
    return false
end

function EditorSDFSubtitleManager:GetSubtitleInfo(subtitleId)
    LOG("[GetSubtitleInfo]")
    local info = self:_GetSubtitleInfo(subtitleId)
    if info then
        likeapp.SDFInfo:SetSDFSubtitleInfo(subtitleId, info)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:UpdateSubtitleFontPath(subtitleId, ttfPath)
    ERROR("[UpdateSubtitleFontPath]" .. tostring(ttfPath))
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:UpdateSubtitleFontPath(ttfPath)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:UpdateFontColor(subtitleId, rgba, useGradient)
    LOG("[UpdateFontColor]")
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:UpdateFontColor(rgba, useGradient)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:SetSubtitleFontPosition(subtitleId, screenPos)
    LOG("[SetSubtitleFontPosition]")
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:SetSubtitleFontPosition(screenPos)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:SetSubtitleFontRotation(subtitleId, rotationZ)
    LOG("[SetSubtitleFontRotation]")
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:SetSubtitleFontRotation(rotationZ)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:SetSubtitleFontScale(subtitleId, scale, isFinalize)
    LOG("[SetSubtitleFontScale]")
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:SetSubtitleFontScale(scale, isFinalize)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:UpdateSubtitleTimeRange(subtitleId, startTs, duration)
    ERROR("[UpdateSubtitleTimeRange] " .. tostring(startTs) .. ", " .. tostring(duration))
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:UpdateSubtitleTimeRange(startTs, duration)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:SetSubtitleFontAlpha(subtitleId, alpha)
    LOG("[SetSubtitleFontAlpha] alpha = " .. tostring(alpha))
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        instance:SetSubtitleFontAlpha(alpha)
        return true
    end
    return false
end

function EditorSDFSubtitleManager:UpdateFrameInfo(frameIndex, pts)
    for subtitleId, node in pairs(self.curNodesMap) do
        local instance = self:_GetNodeBehaviorInstance(subtitleId)
        if instance then
            instance:UpdateFrameInfo(frameIndex, pts)
        end
    end
end

function EditorSDFSubtitleManager:SetTopRenderOrder(subtitleId)
  LOG("[SetTopRenderOrder]" .. tostring(subtitleId))
  local instance = self:_GetNodeBehaviorInstance(subtitleId)
  if instance then
    instance:SetTopRenderOrder();
    return true
  end
  return false
end

function EditorSDFSubtitleManager:SetOriginRenderOrder(subtitleId)
  LOG("[SetOriginRenderOrder]" .. tostring(subtitleId))
  local instance = self:_GetNodeBehaviorInstance(subtitleId)
  if instance then
    instance:SetOriginRenderOrder()
    return true
  end
  return false
end

function EditorSDFSubtitleManager:_GetSubtitleLogicId()
    self.id = self.id + 1
    return self.id
end

function EditorSDFSubtitleManager:_GetMapIndexForId(nodesMap, subtitleId)
    local index = 1
    for id, _ in pairs(nodesMap) do
        if id == subtitleId then
            return index
        else
            index = index + 1
        end
    end
    return 0
end

function EditorSDFSubtitleManager:_IsSubtitleExist(nodesMap, subtitleId)
    return self:_GetMapIndexForId(nodesMap, subtitleId) > 0;
end

function EditorSDFSubtitleManager:_CreateNode(resourcePath)
    self:_CreateRootNodeIfNeed()

    local subtitleNode = self.scene:CreateNode(apollocore.Node.CT_NODE);
    local baseObjects = self:_CreateInstanceFromBundle2(subtitleNode, resourcePath)

    if baseObjects == nil or #baseObjects == 0 then
        return nil
    end

    local scriptComp = subtitleNode:CreateComponent(apolloengine.Node.CT_SCRIPT);
    local scriptPath = "scrs:behavior/sdf_subtitle_node_behavior.lua";
    local instance = self:_LoadBehavior(scriptComp, scriptPath)
    if instance then
        instance:initBehaviorData(subtitleNode, baseObjects, resourcePath);
    end
    self.rootNode:AttachNode(subtitleNode);

    self:_CameraInit(baseObjects);

    self:_ConfigRenderTarget(subtitleNode, self.cameraRenderTarget)

    return subtitleNode
end

function EditorSDFSubtitleManager:_ReleaseNode(node)
    self.rootNode:DetachNode(node);
    self.scene:DeleteNode(node);
end

function EditorSDFSubtitleManager:_CreateRootNodeIfNeed()
    if self.rootNode == nil then
        self.rootNode = self.scene:CreateNode(apollocore.Node.CT_NODE);
        self.rootNode:SetName("SDF_Subtitle");
    end
end

function EditorSDFSubtitleManager:_CreateInstanceFromBundle2(subtitleNode, resPath)
    LOG("[_CreateInstanceFromBundle2]")
    local beginTs = venuscore.ITimerSystem:GetTimevalue()

    local rootconfig = venusjson.LaodJsonFile(resPath)
    if rootconfig == nil then
        return nil
    end

    local pathDir = string.match(resPath, "(.+)/[^/]*%.%w+$");
    local rootdir = pathDir.."/";

    venuscore.IFileSystem:SetResourcePath(rootdir);

    local bundlepath = venuscore.IFileSystem:PathAssembly(rootconfig.scene);

    local _,objects = BundleSystem:DeserializeFromPath(bundlepath, BundleSystem.DeserializeMode.Prefab, self.scene);
    local sceneRootNode = self.scene:GetRootNode();

    LOG("[_CreateInstanceFromBundle2] node's count = " .. tostring(#objects))

    for i=1,#objects do
        local nativeNode = objects[i];
        if nativeNode.GetNativeNode then
            nativeNode = objects[i]:GetNativeNode();
        end

        LOG("[_CreateInstanceFromBundle2] node name = "..nativeNode.Name)
        local coms = nativeNode:GetComponentNames()
        if #coms > 0 then
            for i=1,#coms do
                LOG("---- com name: " .. coms[i])
            end
        end

        local nodeRoot = nativeNode:GetRoot();
        LOG("=> root node name = "..nodeRoot.Name)
        if nodeRoot and nodeRoot:GetObjectID() == sceneRootNode:GetObjectID() then
            -- 已经找到root node
            LOG("attach node")
            subtitleNode:AttachNode(nativeNode);
        end
    end

    local cost = venuscore.ITimerSystem:GetTimevalue() - beginTs;
    ERROR("[_CreateInstanceFromBundle2] cost " .. tostring(cost))
    return objects
end

function EditorSDFSubtitleManager:_GetCameraComponent(objects)
    for i=1, #objects do
        local nativeNode = objects[i];
        if nativeNode.GetNativeNode then
            nativeNode = objects[i]:GetNativeNode();
        end

        local com = nativeNode:GetComponent(apolloengine.Node.CT_CAMERA)
        if com ~= nil then
            return com
        end
    end
    return nil
end

function EditorSDFSubtitleManager:_LoadBehavior(scriptComp, scriptPath)
    local instance = venuscore.LoadBehavior(scriptPath);
    if instance then
        scriptComp:InsertInstance(instance, scriptPath);
    else
        ERROR("[_LoadBehavior] load failed. "..scriptPath)
    end
    return instance;
end

function EditorSDFSubtitleManager:_ConfigRenderTarget(node, rt)
    LOG("[_ConfigRenderTarget]")
    if node == nil or rt == nil then
        return
    end

    local scriptComp = node:GetComponent(apolloengine.Node.CT_SCRIPT);
    if scriptComp ~= nil then
        local paraSet = scriptComp.Instances
        for scrKey,scrValue in pairs(paraSet) do
            if scrValue.ConfigRenderTarget then
                scrValue:ConfigRenderTarget(rt);
            else
                ERROR("ConfigRenderTarget is nil")
            end
        end
    else
        ERROR("scriptComp is nil")
    end
end

function EditorSDFSubtitleManager:_CameraInit(baseobject)
    self.resourceType = self.resourceType + 1
    local sequenceBase = self.sequenceBase
    local sequenceMax = 0
    for i=1,#baseobject do
        local cameracom = baseobject[i]:GetComponent(apolloengine.Node.CT_CAMERA);
        if cameracom then
            cameracom:SetResourceType(self.resourceType);
            local sequence = cameracom:GetSequence() + sequenceBase;
            if sequence > sequenceMax then
                sequenceMax = sequence
            end
            cameracom:SetSequence(sequence);
            cameracom:SetSequenceCulling(false);
            cameracom:SetClearFlag(0)
        end

        local renderCom = baseobject[i]:GetComponent(apolloengine.Node.CT_RENDER);
        if renderCom then
            renderCom:SetResourceType(self.resourceType);
        end
    end
    self.sequenceBase = sequenceMax + 1
end

function EditorSDFSubtitleManager:_GetNodeBehaviorInstance(subtitleId)
    local node = self.curNodesMap[subtitleId]
    if node == nil or venuscore.isNil(node) then
        return nil
    end
    local scriptComp = node:GetComponent(apolloengine.Node.CT_SCRIPT);
    if scriptComp == nil then
        return nil
    end

    local instances = scriptComp.Instances
    for _, scrValue in pairs(instances) do
        return scrValue
    end
end

function EditorSDFSubtitleManager:_GetSubtitleInfo(subtitleId)
    LOG("[_GetSubtitleInfo]")
    local instance = self:_GetNodeBehaviorInstance(subtitleId)
    if instance then
        return instance:GetSubtitleInfo()
    end
    return nil
end


return EditorSDFSubtitleManager;
