local mathfunction = require "mathfunction"
local apolloengine = require "apolloengine"
local apolloDefine = require "apolloutility.defiend"
local collision = require "behavior.vtuber_behavior.capsulecollision"

local laplaciansolver = {}

function laplaciansolver:init()
  self.links = {  {2,{1,7,23,24,25,26,27,28,29,30}},
                  {3,{1,7,23,24,25,26,27,28,29,30}},
                  {8,{7,1,23,24,25,26,27,28,29,30}},
                  {9,{7,1,23,24,25,26,27,28,29,30}},
                };
                
  self.discons = {{1,2},{2,3},{7,8},{8,9}};  
  self.massarr = {0,1,1,1,1,1,0,1,1,1,1,1};
  self.collcons = nil;
  
  self.bindsphere = {};
  self.iterationcount = 10;
  self.maxlappoint = 4;
  self.usewarmup = false;
end

function laplaciansolver:resetcapsule()
  self.capsules = {};
end

function laplaciansolver:addcapsule(pointa,pointb,radius)
  table.insert(self.capsules,{pointa,pointb,radius});
end

function laplaciansolver:clearbindsphere()
  self.bindsphere = {};
end

function laplaciansolver:addbindsphere(index,parentidx,radius,offset,worldrot)
  local sphere = {index,parentidx,radius,offset,worldrot};
  table.insert(self.bindsphere,sphere);
end

function laplaciansolver:setupnodecollision()
  for i=1,#self.bindsphere do
    local index = self.bindsphere[i][1];
    local parentidx = self.bindsphere[i][2];
    local parentdir = mathfunction.vector3(self.modelpose[index][1]-self.modelpose[parentidx][1],
                                           self.modelpose[index][2]-self.modelpose[parentidx][2],
                                           self.modelpose[index][2]-self.modelpose[parentidx][3]);
    local parentrot = mathfunction.Quaternion();
    parentrot:AxisToAxis(mathfunction.vector3(0,1,0),parentdir);
    parentrot:InverseSelf();
    local localrot = self.bindsphere[i][5]*parentrot;
    self.bindsphere[i][6] = localrot;
  end
end

function laplaciansolver:setuplinks()
  self.resetdistance = {};
  for i=1 ,#self.discons  do
    local first = self.discons[i][1];
    local second = self.discons[i][2];
    local firstpos = mathfunction.vector3(self.modelpose[first][1],self.modelpose[first][2],self.modelpose[first][3]);
    local secondpos = mathfunction.vector3(self.modelpose[second][1],self.modelpose[second][2],self.modelpose[second][3]);  
    local distance = (firstpos-secondpos):Length();
    table.insert(self.resetdistance,distance);
  end
end

function laplaciansolver:setuplaplacian()
  
  for i=1,#self.links do
    local centeridx =  self.links[i][1];
    local center = mathfunction.vector3(self.targetpose[centeridx][1],self.targetpose[centeridx][2],self.targetpose[centeridx][3]);
    local disidxmap = {};
    
    for j=1,#self.links[i][2] do
      local otheridx =  self.links[i][2][j];
      local other = mathfunction.vector3(self.targetpose[otheridx][1],self.targetpose[otheridx][2],self.targetpose[otheridx][3]);
      local disvec = other - center;
      local dis = disvec:Length();
      if dis == 0 then
        dis = 9999999;
      else
        dis = 1/dis;
      end
      table.insert(disidxmap,{dis,otheridx});
    end
    
    table.sort(disidxmap, function(v1,v2) return v1[1]>v2[1] end );
    local total =0;
    for i=1, self.maxlappoint do
      total = total+disidxmap[i][1];
    end
    
    local targetlaplacianvec = mathfunction.vector3(0,0,0);
    for j=1,self.maxlappoint do
      local otheridx =  disidxmap[j][2];
      targetlaplacianvec = targetlaplacianvec +
      ( center-mathfunction.vector3(self.targetpose[otheridx][1],self.targetpose[otheridx][2],self.targetpose[otheridx][3]))* (disidxmap[j][1])/total;
    end
    
    local modellaplacianpos = mathfunction.vector3(0,0,0);
    local modelcenter = mathfunction.vector3(self.modelpose[centeridx][1],self.modelpose[centeridx][2],self.modelpose[centeridx][3]);
    for j=1,self.maxlappoint do
      local otheridx =  disidxmap[j][2];
      modellaplacianpos = modellaplacianpos + 
      ( mathfunction.vector3(self.modelpose[otheridx][1],self.modelpose[otheridx][2],self.modelpose[otheridx][3]))*(disidxmap[j][1])/total;
    end
    self.modelpose[centeridx][1] = modellaplacianpos:x()+targetlaplacianvec:x();
    self.modelpose[centeridx][2] = modellaplacianpos:y()+targetlaplacianvec:y();
    self.modelpose[centeridx][3] = modellaplacianpos:z()+targetlaplacianvec:z();
    
  end
  
end

function laplaciansolver:solve_links()
  local index = 1;
  for i=1 ,#self.discons  do
    local first = self.discons[i][1];
    local second = self.discons[i][2];
    
    local firstpos = mathfunction.vector3(self.modelpose[first][1],self.modelpose[first][2],self.modelpose[first][3]);
    local secondpos = mathfunction.vector3(self.modelpose[second][1],self.modelpose[second][2],self.modelpose[second][3]); 
    
    local distance = (firstpos-secondpos):Length();
    local restdistance = self.resetdistance[index];
    
    local tomove = distance-restdistance;

    local dirfs = secondpos-firstpos;
    dirfs:NormalizeSelf();

    local weightfirst  = self.massarr[first]/( self.massarr[first]+self.massarr[second]);
    local weightsecond = self.massarr[second]/( self.massarr[first]+self.massarr[second]);
    
    local firstmove = dirfs*weightfirst*tomove;
    local secondmove = dirfs*weightsecond*tomove*(-1);
    
    self.modelpose[first][1] = self.modelpose[first][1]+firstmove:x();
    self.modelpose[first][2] = self.modelpose[first][2]+firstmove:y();
    self.modelpose[first][3] = self.modelpose[first][3]+firstmove:z();
    
    self.modelpose[second][1] = self.modelpose[second][1]+secondmove:x();
    self.modelpose[second][2] = self.modelpose[second][2]+secondmove:y();
    self.modelpose[second][3] = self.modelpose[second][3]+secondmove:z();
    index = index+1;
  end
end

function laplaciansolver:solve_collision()
  local index = 1;
  for i=1 ,#self.capsules  do
    for j=1,#self.bindsphere do
      local index = self.bindsphere[j][1];
      local parentidx = self.bindsphere[j][2];
      local sphereradius = self.bindsphere[j][3]
      local parentdir = mathfunction.vector3(self.modelpose[index][1]-self.modelpose[parentidx][1],
                                             self.modelpose[index][2]-self.modelpose[parentidx][2],
                                             self.modelpose[index][2]-self.modelpose[parentidx][3]);
      local parentrot = mathfunction.Quaternion();
      parentrot:AxisToAxis(mathfunction.vector3(0,1,0),parentdir);
      local worldrot = self.bindsphere[j][6]*parentrot;
      local pos = mathfunction.vector3(self.modelpose[index][1],self.modelpose[index][2],self.modelpose[index][3]);
      pos = pos +  self.bindsphere[j][4]*worldrot;
      local collided ,offset = collision:collide(pos,sphereradius,self.capsules[i][1],self.capsules[i][2],self.capsules[i][3]);
      if collided then
        self.modelpose[index][1] = self.modelpose[index][1]+offset:x();
        self.modelpose[index][2] = self.modelpose[index][2]+offset:y();
        self.modelpose[index][3] = self.modelpose[index][3]+offset:z();
      end
    end
  end
end

function laplaciansolver:solve(targetpos,modelpos)
 
  self.targetpose = targetpos;
  self.modelpose = modelpos;
  self:setuplinks();
  self:setupnodecollision();
  self:setuplaplacian();
  self.premodelpose = self:tbl_copy(modelpos);
  if self.usewarmup then
    self:applypredelta();
  end
  for i=1,self.iterationcount do
    self:solve_links();
    self:solve_collision();
  end
  if self.usewarmup then
    self:getpredelta();
  end
  return modelpos;
end

function laplaciansolver:getpredelta()
  self.predelta = {};
  for i =1,#self.links do
    local idx =  self.links[i][1];
    self.predelta[i] = {idx,{}};
    self.predelta[i][2][1] = self.modelpose[idx][1] - self.premodelpose[idx][1];
    self.predelta[i][2][2] = self.modelpose[idx][2] - self.premodelpose[idx][2];
    self.predelta[i][2][3] = self.modelpose[idx][3] - self.premodelpose[idx][3];
  end
end

function laplaciansolver:applypredelta()
  if self.predelta==nil then 
    return;
  end
  for i =1,#self.predelta do
    local idx =  self.predelta[i][1];
    self.modelpose[idx][1] = self.modelpose[idx][1] +self.predelta[i][2][1]*0.75;
    self.modelpose[idx][2] = self.modelpose[idx][2] +self.predelta[i][2][2]*0.75;
    self.modelpose[idx][3] = self.modelpose[idx][3] +self.predelta[i][2][3]*0.75;
  end
end

function laplaciansolver:tbl_copy(orig)
  local orig_type = type(orig)
  local copy
  if orig_type == "table" then
      copy = {}
      for orig_key, orig_value in next, orig, nil do
          copy[self:tbl_copy(orig_key)] = self:tbl_copy(orig_value)
      end
  else -- number, string, boolean, etc
      copy = orig
  end
  return copy
end

return laplaciansolver;