local venuscore = require "venuscore"
local mathfunction = require "mathfunction"
local apolloengine = require "apolloengine"
local apollocore = require "apollocore"
local apollonode = require "apolloutility.apollonode"
local statictexmain = require "posteffect_filter.statictexmain"

local dynamicexmain={};

setmetatable(dynamicexmain, {__index = statictexmain});

function dynamicexmain:Initialize(cameraLayer, config)
  statictexmain.Initialize(self, cameraLayer, config)
  if self.maincamera == nil or 
     self.posteffectlist == nil or
     config == nil or 
     config.animation_control == nil then
    return
  end
  
  self.is_played = false
  
  -- Get the texture mapping [texure_name, posteffect_queueorder]
  self.texture_mapping = self:_GetPostEffectTextureMapping()
  if self.texture_mapping == nil then
    ERROR("There are no textures in the post effects to be replaced")
    return
  end
  
  -- read animation control from config
  local ani_control = config.animation_control
  if ani_control == nil then
    return
  end
  
  self.delay = ani_control.Delay or 0  
  
  self.ani_player = self:_CreateFrameAnimationComponent(ani_control)
end

function dynamicexmain:Update(def)  
  if self.timer == nil then
    self.timer = 0
  else
    self.timer = self.timer + def * 1000
  end

  if self.timer > self.delay and not self.is_played  then
    if self.ani_player ~= nil then
      self.is_played = true
      self.ani_player:Play()
    end
  end
  
  self.ani_player:Update(def)
  statictexmain.Update(self, def)
end

-- create frame animation component based on the config 
function dynamicexmain:_CreateFrameAnimationComponent(ani_config)
  if ani_config == nil or 
     self.maincamera == nil then
    return nil
  end
  
 -- create a frame player component
  local ani_player = self.maincamera:CreateComponent(apolloengine.Node.CT_FRAME_PLAYER)
  if ani_player == nil then
    ERROR("Can not create FrameAnimationComponent.")
    return
  end 
  
  -- set project path in component
  ani_player:Awake()
  
  -- refresh binded cameras
  ani_player:RefreshCameraComponentNames(true)
  
  -- get the camera index in frameanimationcomponent 
  local camera_index = self:_GetCameraIndexInAnimationController(self.maincamera, ani_player)
  if camera_index == nil then
    ERROR("Can not find camera in FrameAnimationComponent.")
    return nil
  end
  
  -- set animation entity
  local animations = ani_config.animations  
  ani_player:ResizeEntity(#animations)
  for i = 1, #animations, 1 do
    local animation_source = animations[i]
    local ani_entity = ani_player:GetAniEntity(i - 1)
    self:_SetAnimationEntity(animation_source, camera_index, ani_entity)
  end
  return ani_player
end

function dynamicexmain:_GetCameraIndexInAnimationController(camera, ani_player)
  if camera == nil or 
     ani_player == nil then
    return nil
  end
  
  local camera_index = nil
  local camera_name = camera.node:GetName()
  local binded_cameras = ani_player:GetBindCameraComponentsNames()
  for i = 1, #binded_cameras do
     if camera_name == binded_cameras[i] then
       camera_index = i - 1
       break
     end
  end
  return camera_index
end

function dynamicexmain:_SetAnimationEntity(ani_source, camera_index, ani_entity)
  if ani_entity == nil or 
     camera_index == nil or 
     ani_source == nil then 
    return
  end
  
  ani_entity.isloop = ani_source.Loop
  ani_entity.speed = ani_source.PlaySpeed
  ani_entity.fps = ani_source.FPS
  ani_entity:SetFullPathManual(venuscore.IFileSystem:PathAssembly(ani_source.ImageFullPath), true)
  local post_order = self.texture_mapping[ani_source.ReplaceTex]
  if post_order  ~= nil then
    ani_entity:AttachPostEffectTexture(camera_index, post_order, ani_source.ReplaceTex)   
    ani_entity:SetPostEffectAttachTextureValue(camera_index, post_order, ani_source.ReplaceTex, true)
  end    
end

-- Get texture mapping 
function dynamicexmain:_GetPostEffectTextureMapping()
  local mapping = {}
  for i = 1, #self.posteffectlist do
    local posteffect = self.posteffectlist[i]
    local params = posteffect.post.Parameters
    for k, v in pairs(params) do
      local vType = type(v);
      if vType ~= 'number' and vType ~= 'string'and vType ~= 'boolean' then
        if v:isType(apollocore.TextureEntity:RTTI()) then
          mapping[k] = i - 1
        end
       end
    end
  end
  return mapping
end

function dynamicexmain:_Clear()
  statictexmain._Clear(self)
  
  if self.maincamera == nil or 
     self.ani_player == nil then
    return
  end
  
  -- cameranode only have CreateComponent...
  self.maincamera.node:DeleteComponent(apolloengine.Node.CT_FRAME_PLAYER)  
end

return dynamicexmain;