local vc = require "venuscore"
local ae = require "apolloengine"
local mf = require "mathfunction"
local Fn = require "functional"
local ObjectStateWrapper = require "object_state_wrapper"


local function stringsToCombo(strings)
  return Fn(strings)
    :Map(
      function(str)
        return
        {
          key = str,
          value = str,
        }
      end
    )
    :Get()
end

------------------
local function ShallShowByCategory(desc, key, value)
  if key == "category" then
    return true
  end
  if key == "root" then
    return false
  end
  if desc.category == "all" or
     desc.category == "any" or
     desc.category == "random" or
     desc.category == "sequential" then
    return key == "list" or key == "toMutexGroup"
  elseif desc.category == "content" then
    return key ~= "list" and key ~= "behavior" and key ~= "toMutexGroup"
  elseif desc.category == "behavior" then
    return key ~= "list" and key ~= "content" and key ~= "toMutexGroup"
  end
end
------------------

local stickerTriggerValueCombo =
{
  {
    key = "Not Care",
    value = nil,
  },
  {
    key = "True",
    value = true,
  },
  {
    key = "False",
    value = false,
  },
};

------------------

local StickerBehaviorBtn = vc.VenusBehavior:extend("StickerBehaviorBtn");

function StickerBehaviorBtn:new(onBtn)
  self.onBtn = onBtn;
end

function StickerBehaviorBtn:OnBtn()
  self.onBtn()
end

-------------------

local StickerTriggerDesc = vc.VenusBehavior:extend("StickerTriggerDesc")

function StickerTriggerDesc:new(root)
  self.root = root
  self.category = "all"
  self.content = nil
  self.behavior = nil
  self.stateful = nil
  self.property = nil
  self.condition = nil
  self.list = {}
end

function StickerTriggerDesc:OnStart()
end

function StickerTriggerDesc:OnStop()
end

function StickerTriggerDesc:Test()
  if self.category == "all" then
    for _, subTrigger in ipairs(self.list) do
      if not subTrigger:Test() then
        return false
      end
    end
    return true
  elseif self.category == "any" then
    for _, subTrigger in ipairs(self.list) do
      if subTrigger:Test() then
        return true
      end
    end
    return false
  else
    if self.stateful == nil then
      local inst = nil
      if self.category == "content" then
        inst = self.content
      elseif self.category == "behavior" then
        inst = self.behavior
      end
      if inst == nil then
        return false
      end
      self.stateful = ObjectStateWrapper(inst)
    end
    local prevValue = self.stateful:GetPrevProp(self.property)
    local currValue = self.stateful:GetCurrProp(self.property)
    local cond = false
    if self.condition == "off-on" then
      cond = (not prevValue) and (currValue)
    elseif self.condition == "on-off" then
      cond = (prevValue) and (not currValue)
    elseif self.condition == "on-on" then
      cond = (prevValue) and (currValue)
    elseif self.condition == "off-off" then
      cond = (not prevValue) and (not currValue)
    elseif self.condition == "on" then
      cond = (currValue)
    elseif self.condition == "off" then
      cond = (not currValue)
    end
    return cond
  end
end

function StickerTriggerDesc:Release()
  if self.stateful ~= nil then
    self.stateful:Unref()
    self.stateful = nil
  end
  for _, subTrigger in ipairs(self.list) do
    subTrigger:Release()
  end
end

--This always-hidden-member is to
--make cloning process correct
StickerTriggerDesc:MemberRegister(
  "root",
  vc.ScriptTypes.ReferenceType(
    vc.VenusBehavior
  )
);

StickerTriggerDesc:MemberRegister(
  "category",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "All",
        value = "all"
      },
      {
        key = "Any",
        value = "any"
      },
      {
        key = "Content",
        value = "content"
      },
      {
        key = "Behavior",
        value = "behavior"
      },
    }
  )
);

StickerTriggerDesc:MemberRegister(
  "content",
  vc.ScriptTypes.ReferenceType(
    ae.IContent:RTTI(),
    nil,
    function(thiz, value)
      thiz.content = value
      if thiz.stateful ~= nil then
        thiz.stateful:Unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.content)
    end
  )
);
StickerTriggerDesc:MemberRegister(
  "behavior",
  vc.ScriptTypes.ReferenceType(
    vc.VenusBehavior,
    nil,
    function(thiz, value)
      thiz.behavior = value
      if thiz.stateful ~= nil then
        thiz.stateful:Unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.behavior)
    end
  )
);
StickerTriggerDesc:MemberRegister(
  "property",
  vc.ScriptTypes.ComboType(
    function(thiz)
      local inst = thiz.content or thiz.behavior
      if inst == nil then
        return {}
      end
      if inst.GetProperties == nil then
        return {}
      end
      local props = inst:GetProperties("boolean")
      return stringsToCombo(props)
    end
  )
);
StickerTriggerDesc:MemberRegister(
  "condition",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "Off->On",
        value = "off-on"
      },
      {
        key = "On->Off",
        value = "on-off"
      },
      {
        key = "On->On",
        value = "on-on"
      },
      {
        key = "Off->Off",
        value = "off-off"
      },
      {
        key = "On",
        value = "on"
      },
      {
        key = "Off",
        value = "off"
      },
    }
  )
);


StickerTriggerDesc:MemberRegister(
  "list",
  vc.ScriptTypes.ArrayType(
    function(thiz)
      return StickerTriggerDesc(thiz.root)
    end,
    function (thiz)
      return { unpack(thiz.list) }
    end,
    function (thiz, value)
      thiz.list = { unpack(value) }
    end
  ),
  {
    itemTag =
    {
      shallShowItem = ShallShowByCategory
    }
  }
);

------------------


local StickerActionDesc = vc.VenusBehavior:extend("StickerActionDesc")

function StickerActionDesc:new(root)
  self.root = root
  self.category = "all"
  self.content = nil
  self.behavior = nil
  self.property = nil
  self.value = nil
  self.list = {}
  self.seqentialCounter = 1
  self.pendingCommands = {}
  local thiz = self
  self.toMutexGroup = StickerBehaviorBtn(
    function()
      thiz:ToMutexGroup()
    end
  )
end

function StickerActionDesc:OnStart()
end

function StickerActionDesc:OnStop()
end

function StickerActionDesc:Take()
  if self.category == "all" then
    for _, subAction in ipairs(self.list) do
      subAction:Take()
    end
  elseif self.category == "random" then
    if #self.list > 0 then
      local r = math.random(#self.list)
      local subAction = self.list[r]
      subAction:Take()
    end
  elseif self.category == "sequential" then
    local subAction = self.list[self.seqentialCounter]
    self.seqentialCounter = self.seqentialCounter + 1
    if self.seqentialCounter > #self.list then
      self.seqentialCounter = 1
    end
    subAction:Take()
  else
    if self.value == nil then
      return
    end
    if self.stateful == nil then
      local inst = nil
      if self.category == "content" then
        inst = self.content
      elseif self.category == "behavior" then
        inst = self.behavior
      end
      if inst == nil then
        return
      end
      self.stateful = ObjectStateWrapper(inst)
    end
    self.stateful:SetNextProp(self.property, self.value)
  end
end

function StickerActionDesc:Release()
  if self.stateful ~= nil then
    self.stateful:Unref()
    self.stateful = nil
  end
  for _, subAction in ipairs(self.list) do
    subAction:Release()
  end
end

function StickerActionDesc:ToMutexGroup()
  self.root:ToMutexGroup(self)
end

--This always-hidden-member is to
--make cloning process correct
StickerActionDesc:MemberRegister(
  "root",
  vc.ScriptTypes.ReferenceType(
    vc.VenusBehavior
  )
);

StickerActionDesc:MemberRegister(
  "category",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "All",
        value = "all"
      },
      {
        key = "Random",
        value = "random"
      },
      {
        key = "Sequential",
        value = "sequential"
      },
      {
        key = "Content",
        value = "content"
      },
      {
        key = "Behavior",
        value = "behavior"
      },
    }
  )
);

StickerActionDesc:MemberRegister(
  "content",
  vc.ScriptTypes.ReferenceType(
    ae.IContent:RTTI(),
    nil,
    function(thiz, value)
      thiz.content = value
      if thiz.stateful ~= nil then
        thiz.stateful:Unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.content)
    end
  )
);
StickerActionDesc:MemberRegister(
  "behavior",
  vc.ScriptTypes.ReferenceType(
    vc.VenusBehavior,
    nil,
    function(thiz, value)
      thiz.behavior = value
      if thiz.stateful ~= nil then
        thiz.stateful:Unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.behavior)
    end
  )
);
StickerActionDesc:MemberRegister(
  "property",
  vc.ScriptTypes.ComboType(
    function(thiz)
      local inst = thiz.content or thiz.behavior
      if inst == nil then
        return {}
      end
      if inst.GetProperties == nil then
        return {}
      end
      local props = inst:GetProperties("boolean")
      return stringsToCombo(props)
    end
  )
);
StickerActionDesc:MemberRegister(
  "value",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "On",
        value = true,
      },
      {
        key = "Off",
        value = false,
      }
    }
  )
);

StickerActionDesc:MemberRegister(
  "list",
  vc.ScriptTypes.ArrayType(
    function(thiz)
      return StickerActionDesc(thiz.root)
    end,
    function (thiz)
      return { unpack(thiz.list) }
    end,
    function (thiz, value)
      thiz.list = { unpack(value) }
    end
  ),
  {
    itemTag = 
    {
      shallShowItem = ShallShowByCategory
    }
  }
);

StickerActionDesc:MemberRegister(
  "toMutexGroup",
  nil,
  {
    type = "button"
  }
)

------------------

local StickerTransitionDesc = vc.VenusBehavior:extend("StickerTransitionDesc");

function StickerTransitionDesc:new(root)
  self.id = "transition"
  self.trigger = StickerTriggerDesc(root)
  self.action = StickerActionDesc(root)
end

function StickerTransitionDesc:OnStart()
  self.trigger:OnStart()
  self.action:OnStart()
end

function StickerTransitionDesc:OnStop()
  self.trigger:OnStop()
  self.action:OnStop()
end

function StickerTransitionDesc:Process()
  if self.trigger:Test() then
    self.action:Take()
  end
end

function StickerTransitionDesc:Release()
  self.trigger:Release()
  self.action:Release()
end

StickerTransitionDesc:MemberRegister("id");
StickerTransitionDesc:MemberRegister(
  "trigger",
  nil,
  {
    shallShowItem = ShallShowByCategory
  }
);
StickerTransitionDesc:MemberRegister(
  "action",
  nil,
  {
    shallShowItem = ShallShowByCategory
  }
);

------------------

local StickerMutexGroupItem = vc.VenusBehavior:extend("StickerMutexGroupItem")

function StickerMutexGroupItem:new()
  self.category = "content"
  self.content = nil
  self.behavior = nil
  self.property = nil
  self.stateful = nil
end

function StickerMutexGroupItem:GetTarget()
  if self.stateful == nil then
    local inst = nil
    if self.category == "content" then
      inst = self.content
    elseif self.category == "behavior" then
      inst = self.behavior
    end
    if inst ~= nil then
      self.stateful = ObjectStateWrapper(inst)
    end
  end
  return self.stateful
end

function StickerMutexGroupItem:Release()
  if self.stateful ~= nil then
    self.stateful:Unref()
    self.stateful = nil
  end
end

StickerMutexGroupItem:MemberRegister(
  "category",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "Content",
        value = "content",
      },
      {
        key = "Behavior",
        value = "behavior",
      },
    }
  )
);

StickerMutexGroupItem:MemberRegister(
  "content",
  vc.ScriptTypes.ReferenceType(
    ae.IContent:RTTI(),
    nil,
    function (thiz, value)
      thiz.behavior = nil
      thiz.content = value
      if thiz.stateful ~= nil then
        thiz.stateful:unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.content)
    end
  )
);

StickerMutexGroupItem:MemberRegister(
  "behavior",
  vc.ScriptTypes.ReferenceType(
    vc.VenusBehavior,
    nil,
    function (thiz, value)
      thiz.content = nil
      thiz.behavior = value
      if thiz.stateful ~= nil then
        thiz.stateful:unref()
      end
      thiz.stateful = ObjectStateWrapper(thiz.behavior)
    end
  )
);

StickerMutexGroupItem:MemberRegister(
  "property",
  vc.ScriptTypes.ComboType(
    function(thiz)
      local inst = nil
      if thiz.category == "content" then
        inst = thiz.content
      elseif thiz.category == "behavior" then
        inst = thiz.behavior
      end
      if inst == nil then
        return {}
      end
      if inst.GetProperties == nil then
        return {}
      end
      local props = inst:GetProperties("boolean")
      return stringsToCombo(props)
    end
  )
);

------------------

local StickerMutexGroup = vc.VenusBehavior:extend("StickerMutexGroup");

function StickerMutexGroup:new()
  self.group = {}
end

function StickerMutexGroup:Process()
  local toBeActivated = nil
  for _, item in ipairs(self.group) do
    local stateful = item:GetTarget()
    if stateful ~= nil then
      local nextValue = stateful:GetNextProp(item.property)
      if nextValue then
        toBeActivated = stateful
        break
      end
    end
  end
  if toBeActivated == nil then
    return
  end
  for _, item in ipairs(self.group) do
    local stateful = item:GetTarget()
    if stateful ~= nil and stateful ~= toBeActivated then
      stateful:SetNextProp(item.property, false)
    end
  end
end

function StickerMutexGroup:Release()
  for _, item in ipairs(self.group) do
    item:Release()
  end
end

StickerMutexGroup:MemberRegister(
  "group",
  vc.ScriptTypes.ArrayType(
    function (thiz)
      return StickerMutexGroupItem()
    end,
    function (thiz)
      return { unpack(thiz.group) }
    end,
    function (thiz, value)
      thiz.group = { unpack(value) }
    end
  ),
  {
    itemTag = 
    {
      shallShowItem = ShallShowByCategory
    }
  }
);

-----------------

local StickerBehaviorDesc = vc.VenusBehavior:extend("StickerBehaviorDesc");

function StickerBehaviorDesc:new()
  self.version = 1
  self.title = ""
  self.transitions = {}
  self.mutexGroups = {}
  self.state = "stopped"
end

StickerBehaviorDesc:MemberRegister("version", nil, "dragInt");
StickerBehaviorDesc:MemberRegister("title");
StickerBehaviorDesc:MemberRegister(
  "transitions",
  vc.ScriptTypes.ArrayType(
    function(thiz)
      return StickerTransitionDesc(thiz)
    end,
    function (thiz)
      return { unpack(thiz.transitions) }
    end,
    function (thiz, value)
      thiz.transitions = { unpack(value) }
    end
  )
);
StickerBehaviorDesc:MemberRegister(
  "mutexGroups",
  vc.ScriptTypes.ArrayType(
    function(thiz)
      return StickerMutexGroup()
    end,
    function (thiz)
      return { unpack(thiz.mutexGroups) }
    end,
    function (thiz, value)
      thiz.mutexGroups = { unpack(value) }
    end
  ),
  {
    type = "defaultClosed"
  }
);

StickerBehaviorDesc:MemberRegister(
  "state",
  vc.ScriptTypes.ComboType(
    {
      {
        key = "Start",
        value = "started"
      },
      {
        key = "Stop",
        value = "stopped"
      },
    },
    nil,
    function(thiz, value)
      local oldValue = thiz.state
      thiz.state = value
      if oldValue == "stopped" and value == "started" then
        thiz:OnStart()
      elseif oldValue == "started" and value == "stopped" then
        thiz:OnStop()
      end
    end
  )
);

function StickerBehaviorDesc:OnStart()
  LOG("SUNTYLOG: OnStart " .. self:GetObjectID())
  for _, trans in ipairs(self.transitions) do
    trans:OnStart()
  end
end
function StickerBehaviorDesc:OnStop()
  LOG("SUNTYLOG: OnStop " .. self:GetObjectID())
  for _, trans in ipairs(self.transitions) do
    trans:OnStop()
  end
end

function StickerBehaviorDesc:_OnUpdate(delta)
  if self.state ~= "started" then
    return
  end
  for _, trans in ipairs(self.transitions) do
    trans:Process()
  end
  for _, mutexGroup in ipairs(self.mutexGroups) do
    mutexGroup:Process()
  end
end

function StickerBehaviorDesc:_OnDestroy()
  for _, trans in ipairs(self.transitions) do
    trans:Release()
  end
  for _, mutexGroup in ipairs(self.mutexGroups) do
    mutexGroup:Release()
  end
end

function StickerBehaviorDesc:ToMutexGroup(actionDesc)
  local index = #self.mutexGroups + 1
  self.mutexGroups[index] = StickerMutexGroup()

  local com = self.Node:GetComponent(ae.Node.CT_SCRIPT)
  local itemIndex = 0
  for i = 1, #actionDesc.list do
    local subAction = actionDesc.list[i]
    if subAction.category ~= "content" and subAction.category ~= "behavior" then
      goto CONTINUE
    end
    itemIndex = itemIndex + 1
    self.mutexGroups[index].group[itemIndex] = StickerMutexGroupItem()
    self.mutexGroups[index].group[itemIndex].category = subAction.category
    if subAction.category == "content" then
      self.mutexGroups[index].group[itemIndex].content = subAction.content
    elseif subAction.category == "behavior" then
      self.mutexGroups[index].group[itemIndex].behavior = subAction.behavior
    end
    self.mutexGroups[index].group[itemIndex].property = subAction.property
    ::CONTINUE::
  end
end

return StickerBehaviorDesc;