local Types = require "venuscore.rtti.types"
local libvenuscore = require "libvenuscore"
local AE = require "apolloengine"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"

local BlueDynamicNode = require "bluecore.base.blue_dynamic_node"
local GetVariableRef = BlueDynamicNode:extend();

GetVariableRef:MemberRegister("refComp", Types.ReferenceType(AE.BlueprintComponent:RTTI()));
GetVariableRef:MemberRegister("isLocal");
GetVariableRef:MemberRegister("varName");


function GetVariableRef:new(graph)

  GetVariableRef.super.new(self, graph)

  self.varName = ""
  self.refComp = nil
  self.isLocal = nil -- if isLocal, refBlueprint is BlueFunction, otherwise, BluePrint
  self.refBluegraph = nil

end

function GetVariableRef:IsAllowDuplicate()
  return self.refBluegraph ~= nil and self.refBluegraph:GetVariableInfo(self.varName) ~= nil
end

function GetVariableRef:_OnDuplicate()

  local luaPath = self[BD.LUA_CLASS_PATH];
  local clazz = require(luaPath);
  local bluePrintNode = clazz(self.graph);
  bluePrintNode:EditorCreate(self.refComp, self.varName, self.isLocal);
  bluePrintNode[BD.LUA_CLASS_PATH] = luaPath;
  return bluePrintNode;

end


function GetVariableRef:_OnDeserializePost()

  GetVariableRef.super._OnDeserializePost(self);

  if not self.isLocal and self.refComp == nil then error ("GetVariableRef:_OnDeserializePost refComp is nil") end

  self:_Setup()

end


function GetVariableRef:Update(args)

end

function GetVariableRef:GetOutputByIndex(index)
  return self.refBluegraph[self.varName]
end


function GetVariableRef:_OnCompile()

  self.compileResult = nil

  if self.varName == nil then
    self.compileResult = "serialization fail";
    return false
  end

  if not self.isLocal and libvenuscore.isNil(self.refComp)  then
    self.compileResult = "component is lost";
    return false
  end

  if not self:_Setup() then
    return false
  end

  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  if not varInfo then
    self.compileResult = string.format("variable %s is lost", tostring(self.varName) );
    ERROR("GetVariableRef "..self.compileResult)
    return false
  end

  local rtti = varInfo:GetRtti();
  local pinRtti = self[BD.OUTPUTS_INFO][1]:GetRtti();

  if not rtti:isType(pinRtti) then -- 如果编译时候不一致 需要打开蓝图同步
    self.compileResult = string.format("get variable %s bluenode inconsistent with variable, open and check it", tostring(self.varName));
    ERROR("GetVariableRef "..self.compileResult)
    return false ;
  end

  return true ;
end

function GetVariableRef:SyncEd()

  if not self.isLocal and libvenuscore.isNil(self.refComp)  then
    self.compileResult = "component is lost";
    ERROR("GetVariableRef "..self.compileResult);
    return
  end

  if not self:_Setup() then
    return false
  end

  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  if not varInfo then
    self.compileResult = "varInfo "..tostring(self.varName).." is lost";
    ERROR("GetVariableRef "..self.compileResult);
    return
  end

  local hostNodeName
  if self.isLocal then
    --local blueFunction =  self.graph ;
    hostNodeName = "function"
  else
    hostNodeName = self.refComp:GetHostNode():GetName()
  end
  local rtti = varInfo:GetRtti();
  local displayName = varInfo:GetDisplayName()
  
  assert(rtti, "varInfo rtti nil")
  
  self[BD.OUTPUTS_INFO][1]:ChangeRtti(rtti, nil)
  self[BD.OUTPUTS_INFO][1]:ChangeNameAndHint(displayName.."\n in "..hostNodeName, "Reference of "..displayName.." in ".. hostNodeName)

end

function  GetVariableRef:_Setup()
  if self.isLocal then
    self.refBluegraph = self.graph;
    -- 如果是局部变量,那么get节点和变量一定在同一个蓝图中的
  else
    local blueprint = BU.GetIns(self.refComp)
    if blueprint == nil then
      self.compileResult = "blueprint is deleted"
      ERROR("GetVariableRef "..self.compileResult);
      return false
    end
    self.refBluegraph =  blueprint;
  end

  if not self.refBluegraph then
    self.compileResult = "blueprint/bluefunction is deleted"
    ERROR("GetVariableRef "..self.compileResult);
    return false
  end
  return true
end

function GetVariableRef:GetVariableName()
  return self.varName -- BlueVariableInfo.name is Id name
end

function GetVariableRef:GetRefBluegraph()
  return self.refBluegraph
end

function GetVariableRef:IsVarArrayEd()
  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  return varInfo:IsArray();
end

-- -- 编辑器运行时
if _KRATOSEDITOR then

  function GetVariableRef:_OnCreateWithEditor(refBlueprintComp, varName, isLocal)

    GetVariableRef.super._OnCreateWithEditor(self);

    assert(isLocal or refBlueprintComp, "[GetVariableRef] refComp is nil ")
    assert(varName, "[GetVariableRef] varIdx is nil ")

    self.refComp = refBlueprintComp
    self.varName = varName
    self.isLocal = isLocal -- if true, self.refComp is null

    if self:_Setup() then

      local varInfo = self.refBluegraph:GetVariableInfo(varName);

      local rtti = varInfo:GetRtti()
      local displayName = varInfo:GetDisplayName();

      local hostNodeName
      if self.isLocal then
        --local blueFunction =  self.graph ;
        hostNodeName = "function" -- for local variable, it's function name
      else
        hostNodeName = self.refComp:GetHostNode():GetName()
      end

      self[BD.OUTPUTS_INFO][1]:ChangeRtti(rtti, nil)
      self[BD.OUTPUTS_INFO][1]:ChangeNameAndHint(displayName.."\n in "..hostNodeName, "Reference of "..displayName.." in ".. hostNodeName)
    else
      return  ;
    end
  end

end

GetVariableRef:SetFunctionType(BD.VARIABLE_NODE);
GetVariableRef:SetFunctionName("ref_variable")
GetVariableRef:RegisterOutput(1, Types.AnyType, "n/a in n/a", "Reference of n/a in n/a", Types.AnyType(), false, nil);


return GetVariableRef