

#include "common.inc"
#include "shadow.inc"

#ifndef LIGHTPBR_INC
#define LIGHTPBR_INC

float3 RotateAroundYInDegrees (float3 vertex, float degrees)
{
	const float PI = 3.14159265359;
	float alpha = degrees * PI / 180.0;
	float sina, cosa;
	sincos(alpha, sina, cosa);
	float2x2 m = float2x2(cosa, -sina, sina, cosa);
	return float3(mul(m, vertex.xz), vertex.y).xzy;
}
float4x4 Euler2Matrix(float heading, float attitude, float bank) 
{
    // Assuming the angles are in radians.
    float ch = cos(heading);
    float sh = sin(heading);
    float ca = cos(attitude);
    float sa = sin(attitude);
    float cb = cos(bank);
    float sb = sin(bank);
    
    
    float4x4 rotM = { ch * ca         , sa,    -sh*ca, 0.0,
    									sh*sb - ch*sa*cb, ca*cb, sh*sa*cb + ch*sb, 0.0,
    									ch*sa*sb + sh*cb, -ca*sb, -sh*sa*sb + ch*cb, 0.0,
    									0.0f,              0.0,    0.0,               1.0
    };
    
    return rotM;
  }

float3 SpecularIBLColor(samplerCUBE cubeTex, float3 worldPos, float3 worldNormal, float roughness, float rotation)
{
	float3 reflectDir = WorldReflectionVector(worldPos, worldNormal);
	float mipmapLevel = roughness * 6.0;
	float3 dir = RotateAroundYInDegrees(reflectDir, rotation);
	float4 rgbm = SAMPLE_TEXCUBE_LOD(cubeTex, dir, mipmapLevel);
	rgbm.xyz = rgbm.xyz * rgbm.w * 6.0;

	return rgbm.xyz;
}

float3 SpecularIBLColor(samplerCUBE cubeTex, float3 worldPos, float3 worldNormal, float roughness)
{
  float3 reflectDir = WorldReflectionVector(worldPos, worldNormal);
  float mipmapLevel = roughness * 6.0;
  float3 dir = RotateAroundYInDegrees(reflectDir, 0);
  float4 rgbm = SAMPLE_TEXCUBE_LOD(cubeTex, dir, mipmapLevel);
  rgbm.xyz = rgbm.xyz * rgbm.w * 6.0;

  return rgbm.xyz;
}

float3 SpecularIBLColorWithRot(samplerCUBE cubeTex, float3 worldPos, float3 worldNormal, float roughness, float4x4 rotM)
{
	float3 reflectDir = WorldReflectionVector(worldPos, worldNormal);
	float mipmapLevel = roughness * 6.0;
	
	reflectDir = mul(rotM, float4(reflectDir, 0.0));
	reflectDir = normalize(reflectDir);
	
	float4 rgbm = SAMPLE_TEXCUBE_LOD(cubeTex, reflectDir, mipmapLevel);
	rgbm.xyz = rgbm.xyz * rgbm.w * 6.0;

	return rgbm.xyz;
}

float3 DiffuseIBLColor(samplerCUBE cubeTex, float3 worldPos, float3 worldNormal)
{
	//float3 reflectDir = WorldReflectionVector(worldPos, worldNormal);
	float4 rgbm = SAMPLE_TEXCUBE(cubeTex, worldNormal);
	rgbm.xyz = rgbm.xyz * rgbm.w * 6.0;

	return rgbm.xyz;
}

float2 calcSphericalTexCoordsFromDir(float3 reflDir)
{
	float m = 2.0 * sqrt(reflDir.x * reflDir.x + reflDir.y * reflDir.y + (reflDir.z + 1.0) * (reflDir.z + 1.0));
	float2 reflTexCoord = reflDir.xy / m + 0.5;
	return reflTexCoord;
}

float3 SampleReflection(sampler2D reflectionTex, float modulation, float reflectionIntensity,float3 worldPos, float3 worldNormal)
{
	float3 reflectDir = WorldReflectionVector(worldPos, worldNormal);
	float2 uv = float2(1.0,1.0) - calcSphericalTexCoordsFromDir(normalize(reflectDir));
	float4 reflectionTexSample = tex2D(reflectionTex, uv);
	float3 reflectionColor = reflectionTexSample.rgb;
	
	return reflectionColor * reflectionIntensity;
}

float3 RimHeightLight(float3 rimColor, float rimIdentity, float rimPow, float3 worldPos, float3 worldNormal)
{
	float3 worldCamDir = normalize(CAMERA_WORLDPOSITION.xyz - worldPos.xyz);
	float rim = 1.0 - max(0, dot(worldCamDir, worldNormal));
	float3 rimCol = rimColor.xyz * pow(rim, rimPow) * float3(rimIdentity, rimIdentity, rimIdentity);
	return rimCol;
}


inline half OneMinusReflectivityFromMetallic(half metallic)
{
	half3 DielectricSpec = half3(0.04, 0.04, 0.04);
    half oneMinusDielectricSpec = 1.0 - DielectricSpec.r;
    return oneMinusDielectricSpec - metallic * oneMinusDielectricSpec;
}

inline half3 DiffuseAndSpecularFromMetallic (half3 albedo, half metallic, out half3 specColor, out half oneMinusReflectivity)
{
	half3 DielectricSpec = half3(0.04, 0.04, 0.04);
	specColor = DielectricSpec.rgb * (1.0 - metallic) + albedo.rgb * metallic;
    oneMinusReflectivity = OneMinusReflectivityFromMetallic(metallic);
    return albedo * oneMinusReflectivity;
}

struct SurfaceStandardInput
{
	float3 albedo;
	half metallic; // 0 = non-metal 1 = metal
	half smoothness; // 0 = rough 1 = smooth
	half occlusion; // occlusion
	float3 emission;
	float3 specularIbl;
	float3 diffuseIbl;
	float3 worldNormal;
	float3 worldPos;
	float atten;
	float alpha;
};

struct LightStandardInput
{
	float3 lightColor;
	float3 viewDir;
	float3 lightDir;
	float3 normalDir;
	float3 reflectDir;
	float atten;
	float3 emission;
	float3 specularIbl;
	float3 diffuseIbl;
	float3 albedo;
	half metallic;
	half smoothness;
	half occlusion;
	float alpha;
};

inline float3 safeNormalize(float3 inVec)
{
    float dp3 = max(0.001f, dot(inVec, inVec));
    return inVec * rsqrt(dp3);
}

inline half Pow5 (half x)
{
    return x*x * x*x * x;
}

half DisneyDiffuse(half NdotV, half NdotL, half LdotH, half perceptualRoughness)
{
    half fd90 = 0.5 + 2 * LdotH * LdotH * perceptualRoughness;
    // Two schlick fresnel term
    half lightScatter   = (1 + (fd90 - 1) * Pow5(1 - NdotL));
    half viewScatter    = (1 + (fd90 - 1) * Pow5(1 - NdotV));

    return lightScatter * viewScatter;
}

inline float SmithJointGGXVisibilityTern(float NdotL, float NdotV, float roughness)
{
	float a = roughness;
    float lambdaV = NdotL * (NdotV * (1 - a) + a);
    float lambdaL = NdotV * (NdotL * (1 - a) + a);

	return 0.5f / (lambdaV + lambdaL + 1e-4f);
}

inline float GGXTerm (float NdotH, float roughness)
{
    float a2 = roughness * roughness;
    float d = (NdotH * a2 - NdotH) * NdotH + 1.0f; // 2 mad
    return INV_PI * a2 / (d * d + 1e-7f); // This function is not intended to be running on Mobile,
                                            // therefore epsilon is smaller than what can be represented by half
}

inline half3 FresnelTerm (half3 F0, half cosA)
{
    half t = Pow5 (1 - cosA);   // ala Schlick interpoliation
    return F0 + (1-F0) * t;
}

inline half3 FresnelLerp (half3 F0, half3 F90, half cosA)
{
    half t = Pow5 (1 - cosA);   // ala Schlick interpoliation
	return F0 * (1.0 - t) + F90 * t;
}
inline half3 PreMultiplyAlpha(half3 diffColor,half alpha,half oneMinusReflectivity,out half outModifiedAlpha)
{
	#if defined(ALPHAPREMULTIPLY_ON)
		diffColor *= alpha;
		outModifiedAlpha = 1 - oneMinusReflectivity + alpha * oneMinusReflectivity;
	#else
		outModifiedAlpha = alpha;
	#endif
	return diffColor;
}

half3 EnvBRDFApprox( half3 SpecularColor, half Roughness, half NoV )
{
	const half4 c0 = { -1, -0.0275, -0.572, 0.022 };
	const half4 c1 = { 1, 0.0425, 1.04, -0.04 };
	half4 r = Roughness * c0 + c1;
	half a004 = min( r.x * r.x, exp2( -9.28 * NoV ) ) * r.x + r.y;
	half2 AB = half2( -1.04, 1.04 ) * a004 + r.zw;

	AB.y *= saturate( 50.0 * SpecularColor.g );
	return SpecularColor * AB.x + AB.y;
}

float GGX_Mobile(float Roughness, float NoH, float3 H, float3 N)
{
    float OneMinusNoHSqr = 1.0 - NoH * NoH;

	float a = Roughness * Roughness;
	float n = NoH * a;
	float p = a / (OneMinusNoHSqr + n * n);
	float d = p * p;
	return d;

	/*
	float3 nxh = cross(N, H);
	float OneMinusNoHSqr = dot(nxh, nxh);
	float a = Roughness * Roughness;
	float n = NoH * a;
	float p = a / (OneMinusNoHSqr + n * n);
	float d = p * p;
	return min(d, 65504.0);
	*/
}

half CalcSpecular(half Roughness, half RoughnessWithClamp, float RoL, float NoH, float3 H, float3 N)
{
	return (Roughness*0.25 + 0.25) * GGX_Mobile(RoughnessWithClamp, NoH, H, N);
}

float3 PBR_Lighting(LightStandardInput input)
{
#ifdef RefUnity
	half3 specColor = half3(0.0, 0.0, 0.0);
	half oneMinusReflectivity = 0.0;

	half3 albedo = DiffuseAndSpecularFromMetallic(input.albedo, input.metallic, specColor, oneMinusReflectivity);
	input.alpha = 0.0;
	albedo = PreMultiplyAlpha(albedo,input.alpha,oneMinusReflectivity, input.alpha);


	float perceptualRoughness = 1.0 - input.smoothness;
	float3 halfDir = safeNormalize(input.lightDir + input.viewDir);
    float dotNV = abs(dot(input.viewDir, input.normalDir));
	float dotNL = saturate(dot(input.normalDir, input.lightDir));
	float dotNH = saturate(dot(input.normalDir, halfDir));

	half dotLV = saturate(dot(input.lightDir, input.viewDir));
	half dotLH = saturate(dot(input.lightDir, halfDir));
	half diffuseTerm = DisneyDiffuse(dotNV, dotNL, dotLH, perceptualRoughness) * dotNL;
	
	float roughness = perceptualRoughness * perceptualRoughness;
	roughness = max(roughness, 0.002);
	float G = SmithJointGGXVisibilityTern(dotNL, dotNV, roughness);
	float D = GGXTerm(dotNH, roughness);

	float specularTerm = D * G * 3.14159265;
    specularTerm = max(0.0, specularTerm * dotNL);

	half surfaceReduction = 0.0;
	surfaceReduction = 1.0 / (roughness*roughness + 1.0);

    half grazingTerm = saturate(input.smoothness + (1-oneMinusReflectivity));

	float3 color = albedo * input.lightColor * diffuseTerm + specularTerm * input.lightColor * FresnelTerm(specColor, dotLH);
	color *= input.atten;
	color += input.emission;//* input.albedo;
	color += albedo * input.diffuseIbl;

	float3 specularEnv = surfaceReduction * FresnelLerp (specColor, grazingTerm.xxx, dotNV);
    color += specularEnv * input.specularIbl;
#else
	float3 reflectDir = input.reflectDir;
	float3 halfDir = normalize(input.lightDir + input.viewDir);
	float dotNV = max(dot(input.viewDir, input.normalDir), 0.0);
	float dotNL = saturate(dot(input.normalDir, input.lightDir));
	float dotRL = max(0, dot(reflectDir, input.lightDir));
	float dotNH = saturate(dot(input.normalDir, halfDir));

	// diff color
	float3 diffColor = input.albedo.xyz - input.albedo.xyz * input.metallic;

	//cal specular
	
	float3 dielectricSpec = float3(0.04, 0.04, 0.04);
	float roughness = 1.0 - input.smoothness;
	
	float roughnessWithClamp = max(0.12, roughness);
	float dFactor = CalcSpecular(roughness, roughnessWithClamp, dotRL, dotNH, halfDir, input.normalDir);

	float3 specColor = (dielectricSpec - dielectricSpec * input.metallic) + input.albedo.xyz;
	specColor = EnvBRDFApprox(specColor, roughness, dotNV);

	// pre multiy alpha
	float oneMinusReflectivity = OneMinusReflectivityFromMetallic(input.metallic);
	input.alpha = 0.0;
	diffColor = PreMultiplyAlpha(diffColor, input.alpha,oneMinusReflectivity, input.alpha);

	// out color
	float3 color = dotNL * input.lightColor *  (diffColor + specColor * dFactor);
	color += diffColor * input.diffuseIbl;
	color += input.specularIbl * specColor;
	color += input.emission;
#endif

    return color;
}
// worldNorm
#ifdef PointLight
float3 PBR_LightCalc(SurfaceStandardInput input)
{
	float3 lightDir = LIGHT_POSITION.xyz - input.worldPos.xyz;
	float dis = sqrt(dot(lightDir.xyz, lightDir.xyz));
	float disAtten = clamp(dis * LIGHT_RANGE_INV, 0.0, 1.0);
	float attenation = (1.0 - disAtten) /  ( LIGHT_ATTENUATION.x + disAtten * LIGHT_ATTENUATION.y + disAtten * disAtten * LIGHT_ATTENUATION.z );
	float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);
	lightDir.xyz = normalize(lightDir.xyz);

	LightStandardInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.metallic = input.metallic;
	lightParam.smoothness = input.smoothness;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.reflectDir.xyz = reflect(-lightParam.viewDir, lightParam.normalDir);
	lightParam.atten = attenation * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;
	lightParam.emission = input.emission;
	lightParam.specularIbl = input.specularIbl;
	lightParam.diffuseIbl = input.diffuseIbl;
	lightParam.alpha = input.alpha;
	
	return PBR_Lighting(lightParam);
}
#elif DirLight
float3 PBR_LightCalc(SurfaceStandardInput input)
{
	float3 lightDir = -LIGHT_GIVEN_DIRECTION.xyz;
	float atten = 1.0;
    float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);

	LightStandardInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.metallic = input.metallic;
	lightParam.smoothness = input.smoothness;
	lightParam.emission = input.emission.xyz;
	lightParam.specularIbl = input.specularIbl;
	lightParam.diffuseIbl = input.diffuseIbl;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.reflectDir.xyz = reflect(-lightParam.viewDir, lightParam.normalDir);
	lightParam.atten = atten * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;
	lightParam.alpha = input.alpha;
	
	return PBR_Lighting(lightParam);
}
#elif SpotLight
float3 PBR_LightCalc(SurfaceStandardInput input)
{
	float3 lightDir = LIGHT_POSITION.xyz - input.worldPos.xyz;
	float dis = sqrt(dot(lightDir.xyz, lightDir.xyz));
	float disAtten = clamp(dis * LIGHT_RANGE_INV, 0.0, 1.0);
	float attenation = (1.0 - disAtten) /  ( LIGHT_ATTENUATION.x + disAtten * LIGHT_ATTENUATION.y + disAtten * disAtten * LIGHT_ATTENUATION.z );
	float attenAngle = 1.0;

	lightDir.xyz = normalize(lightDir.xyz);
	attenAngle = clamp( 1.0 - ( LIGHT_INNER_DIFF_INV.x - dot(lightDir.xyz, -LIGHT_GIVEN_DIRECTION.xyz) ) * LIGHT_INNER_DIFF_INV.y, 0.0, 1.0 );
	attenation *= attenAngle;

	float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);

	LightStandardInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.metallic = input.metallic;
	lightParam.smoothness = input.smoothness;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.emission = input.emission.xyz;
	lightParam.specularIbl = input.specularIbl;
	lightParam.diffuseIbl = input.diffuseIbl;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.reflectDir.xyz = reflect(-lightParam.viewDir, lightParam.normalDir);
	lightParam.atten = attenation * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;
	lightParam.alpha = input.alpha;
	
	return PBR_Lighting(lightParam);
}
#elif NoLight
float3 PBR_LightCalc(SurfaceStandardInput input)
{
    float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);

	LightStandardInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.metallic = input.metallic;
	lightParam.smoothness = input.smoothness;
	lightParam.emission = input.emission.xyz;
	lightParam.specularIbl = input.specularIbl;
	lightParam.diffuseIbl = input.diffuseIbl;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.viewDir.xyz = worldViewDir;
	lightParam.lightDir.xyz = float3(0.0, 0.0, 1.0);
	lightParam.reflectDir.xyz = float3(0.0, 0.0, 1.0);
	lightParam.atten = 1.0;
	lightParam.lightColor = float3(0.0, 0.0, 0.0);
	lightParam.alpha = input.alpha;
	
	return PBR_Lighting(lightParam);
}
#endif

#endif
