local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local mathfunction  = require "mathfunction"
local defined       = require "facemorph.defined"
local venuscore     = require "venuscore"
local likeapp       = require "likeapp"
local warppass      = require "facemorph.warppass"
local morphpass     = require "facemorph.morphpass"

local MorphRendererV2 = Object:extend();

local Stages = 
{
  WARP = "warp",
  MORPH = "morph"
};

function MorphRendererV2:new(morphtime, showtime)  
  self.curTexid = {};
  
  self.imagelist = {};
  self.landmarklist = {};
  self.indiceslist = {};
  self.interval = {};  
  
  self.defaultMorphInterval = morphtime;
  self.defaultShowInterval = showtime;
  
  -- shader passes
  self.passes = {};

  self.passes[Stages.WARP] = warppass();
  self.passes[Stages.MORPH] = morphpass();
end

function MorphRendererV2:Clear()  
  self.curTexid = {};
  
  self.imagelist = {};
  self.landmarklist = {};
  self.indiceslist = {};
  self.interval = {}; 
end

function MorphRendererV2:ReleaseResource()
  self:Clear();
  
  for _, pass in pairs(self.passes)
  do
    pass:ReleaseResource();
    pass = nil;
  end
end

-- InitMorph
-- Note: if use mockup data to do internal test, please use SetMorphTextureInfos to set data
--       comment self:Clear(); then call this func
function MorphRendererV2:InitMorph(indiceslist, showtimelist, transtimelist)   
  self:Clear();
  
  if ( indiceslist == nil or showtimelist == nil or transtimelist == nil)
  then
    return false;
  end    
  
  self.indiceslist = indiceslist;
  
  -- start
  self.interval[1] = 0;    
  for i = 1, #self.indiceslist 
  do
    local showtime = showtimelist[i] or self.defaultShowInterval;
    local transtime = transtimelist[i] or self.defaultMorphInterval;
    self.interval[2*i] = self.interval[2*i-1] + showtime;
    -- do not add the last transition time to the interval list
    if (i ~= self.indiceslist)
    then
      self.interval[2*i + 1] = self.interval[2*i] + transtime;    
    end
  end
  return true;
end

-- Create FBO context based on input fbosize
function MorphRendererV2:SetFboSize(fbosize)
  for _, pass in pairs(self.passes)
  do
    pass:SetFboSize(fbosize);
  end
end

-- Update
function MorphRendererV2:Update(def)
  local leftI, rightI = self:_Search(def);  
  
  if (leftI == rightI)
  then
    self:_DoExhibition(leftI);
  else
    local _start = self.interval[2 * leftI];
    local _end = self.interval[2 * rightI - 1];
    local alpha = (def - _start) / (_end - _start);
    self:_DoMorphing(leftI, rightI, alpha);
    return;
  end
  
  self:SetShow(true);
end

-- Setshow 
function MorphRendererV2:SetShow(show)
  for _, pass in pairs(self.passes)
  do
    pass:SetShow(show);
  end
end

function MorphRendererV2:GetMorphTex()
  if self.passes[Stages.MORPH] == nil then
    return nil;
  else
    return self.passes[Stages.MORPH]:GetTargetTexture();
 end
end

-- Generate texture based on tex path
function MorphRendererV2:LoadTexture(tex_path)
  if (tex_path == nil)
  then
    return nil;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureFileMetadata (
                         apolloengine.TextureEntity.TU_STATIC,
                         apolloengine.TextureEntity.PF_AUTO,
                         1, false,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TF_LINEAR,
                         apolloengine.TextureEntity.TF_LINEAR,
                         tex_path));
     
  texture:SetKeepSource(true); 
  --texture:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);   
  texture:CreateResource(); 
  
  return texture;
end

-- find the morph images in def timeslot
-- return [left, right]
function MorphRendererV2:_Search(def)
  local left, right = 0, #self.interval;
  
  for i = 1, #self.interval 
  do
    local interval = self.interval[i];
    if def >= interval
    then
      left = i;
    else
      right = i;
      break;
    end
  end
  -- Get the index in indiceslist
  left  = math.floor((left + 1 ) / 2);
  right = math.floor((right + 1) / 2);  
  
  return left, right;
end

-- generate texture based on texture stream
function MorphRendererV2:_GenerateTexture(ts)
  if (ts == nil)
  then
    return nil;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureBufferMetadata(
                          apolloengine.TextureEntity.TU_WRITE,
                          1, false,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TF_LINEAR,
                          apolloengine.TextureEntity.TF_LINEAR,
                          ts));
                      
  texture:CreateResource();  
  return texture;
end

-- Create RT context
function MorphRendererV2:_CreateRenderTarget(size)      
  --local clear_color = mathfunction.Color(0.0, 0.0, 0.0, 0.0); -- move to camera
  
  local rt = apolloengine.RenderTargetEntity();
  rt:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                    apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                    mathfunction.vector4(0, 0, size:x(), size:y()),
                                                    size)
                  );

  local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
  tex:PushMetadata(apolloengine.TextureBufferMetadata(size,
                                                      apolloengine.TextureEntity.TT_TEXTURE2D,
                                                      apolloengine.TextureEntity.TU_READ,
                                                      apolloengine.TextureEntity.PF_R8G8B8A8,
                                                      1, false,
                                                      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                      apolloengine.TextureEntity.TF_LINEAR,
                                                      apolloengine.TextureEntity.TF_LINEAR));
                  
  
  rt:CreateResource();

  return rt, tex;
end


-- Morphing source image to target image in interval seconds
function MorphRendererV2:_DoMorphing(index1, index2, alpha)  
  if (index1 < 1 or index1 > #self.indiceslist or 
      index2 < 1 or index2 > #self.indiceslist)
  then
    return false;
  end
  
  local indices = {self.indiceslist[index1], self.indiceslist[index2]};
  
  local update = false;  
  
  local warp_pass = self.passes[Stages.WARP];
  local morph_pass = self.passes[Stages.MORPH];
  
  for i = defined.SOURCE, defined.TARGET do 
    local index = indices[i];
    if (index < 1)
    then
      return false;
    end  
    
    if (self.curTexid[i] == nil or self.curTexid[i] ~= index)
    then 
      if self.imagelist[index] == nil
      then
        self.imagelist[index] = likeapp.AI:GetMorphTexture(index);      
      end
      
      if self.landmarklist[index] == nil
      then
        self.landmarklist[index] = likeapp.AI:GetMorphLandmarks(index);
      end
      
      update = true;
      
      local tex = self:_GenerateTexture(self.imagelist[index]);
      for _, pass in pairs(self.passes)
      do 
        pass:ResetBuffer(i, tex, self.imagelist[index]:GetSize(), self.landmarklist[index]);     
      end

      self.curTexid[i] =  indices[i];
    end
  end

  if(index1 ~= index2 and warp_pass ~= nil)
  then
    if update == true then
      warp_pass:Reset();
    else
      warp_pass:Update(alpha);     
      
      for i = defined.SOURCE, defined.TARGET do 
        local updatedTex = warp_pass:GetTargetTexture(i);
        local updatedLmks = warp_pass:GetWarppedLandmarks(i);
        morph_pass:ResetBuffer(i, updatedTex, updatedTex:GetSize(), updatedLmks);
      end    
      
      alpha = math.sin((alpha * 2 - 1) * 3.1415926 * 90.0/180.0) + 1;
      alpha = alpha * 0.5;
    end
  end
  
  morph_pass:Update(alpha);
  return true;
end

-- Show an image
function MorphRendererV2:_DoExhibition(index)
  return self:_DoMorphing(index, index, 0);
end

-- SetMorphTextureInfos, this is an internally lua used API  
function MorphRendererV2:SetMorphTextureInfos(tslist, landmarklist)
  self.imagelist = tslist;
  self.landmarklist = landmarklist;
end

return MorphRendererV2;
