local vc = require "venuscore"
local ae = require "apolloengine"
local mf = require "mathfunction"
local cv = require "computervisionfunction"

local paramTable =
{
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Eye]   =
  {
    stuff = "stuffLE",
    mask = "maskLE",
    maskMat = "maskMatLE",
    refRect1 = "refRect1LE",
    refRect2 = "refRect2LE",
    refRect3 = "refRect3LE",
    refRect4 = "refRect4LE",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Eye]  =
  {
    stuff = "stuffRE",
    mask = "maskRE",
    maskMat = "maskMatRE",
    refRect1 = "refRect1RE",
    refRect2 = "refRect2RE",
    refRect3 = "refRect3RE",
    refRect4 = "refRect4RE",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Brow]  =
  {
    stuff = "stuffLB",
    mask = "maskLB",
    maskMat = "maskMatLB",
    refRect1 = "refRect1LB",
    refRect2 = "refRect2LB",
    refRect3 = "refRect3LB",
    refRect4 = "refRect4LB",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Brow]  =
  {
    stuff = "stuffRB",
    mask = "maskRB",
    maskMat = "maskMatRB",
    refRect1 = "refRect1RB",
    refRect2 = "refRect2RB",
    refRect3 = "refRect3RB",
    refRect4 = "refRect4RB",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Nose]  =
  {
    stuff = "stuffNS",
    mask = "maskNS",
    maskMat = "maskMatNS",
    refRect1 = "refRect1NS",
    refRect2 = "refRect2NS",
    refRect3 = "refRect3NS",
    refRect4 = "refRect4NS",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Mouth]  =
  {
    stuff = "stuffMH",
    mask = "maskMH",
    maskMat = "maskMatMH",
    refRect1 = "refRect1MH",
    refRect2 = "refRect2MH",
    refRect3 = "refRect3MH",
    refRect4 = "refRect4MH",
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_FullFace]  =
  {
    stuff = "stuffFC",
    mask = "maskFC",
    maskMat = "maskMatFC",
    refRect1 = "refRect1FC",
    refRect2 = "refRect2FC",
    refRect3 = "refRect3FC",
    refRect4 = "refRect4FC",
  },
}
--[[
local anchorsTable =
{
  {
    { 37, 38, 0.25 },
    { 44, 43, 2.50 },
    { 37, 38, 0.75 },
    { 44, 43, 3.50 },
  },
  {
    { 82,  5, 0.6 },
    { 82,  5, 0.5 },
    { 82,  5, 0.4 },
    { 82,  8, 0.5 },
  },
  {
    { 94, 15, 0.50 },
    { 93, 16, 0.25 },
    { 92, 17, 0.50 },
    { 93, 16, 0.75 },
  },
  {
    { 83, 27, 0.6 },
    { 83, 27, 0.5 },
    { 83, 27, 0.4 },
    { 83, 24, 0.5 },
  },
}
]]--
local anchorsTable =
{
  { 44, 43, 3.5 },
  { 82,  5, 0.5 },
  { 46, 46, 0.5 },
  { 83, 27, 0.5 },
}

local refRadius = { 44, 45 }

local dtSize = cv.GPUSegmentComponent.mlRectSize
local dtPosition = cv.GPUSegmentComponent.mlRectPosition
local dtPivot = cv.GPUSegmentComponent.mlRectPivot
local dtRoll = cv.GPUSegmentComponent.mlRectRoll;

local SkinStuffBehavior = vc.VenusBehavior:extend("SkinStuffBehavior")

function SkinStuffBehavior:new()
  self.camera = nil
end

function SkinStuffBehavior:_CollectGPUSeg()
  local gpuSegs = {}
  local subNodes = self.Node:GetChildrens()
  for i = 1, #subNodes do
    local subNode = subNodes[i]
    local gpuSeg = subNode:GetComponent(ae.Node.CT_CV_GPUSEGMENT)
    if gpuSeg ~= nil then
      table.insert(gpuSegs, gpuSeg)
    end
  end
  return gpuSegs
end

function SkinStuffBehavior:_ProcessByGPUSeg(postEffect, gpuSeg)
  
  local segType = gpuSeg.Type
  local paramSet = paramTable[segType]
  if paramSet == nil then
    return
  end
  
  local face106 = gpuSeg:Face106P()
  if face106 == nil then
    return
  end

  local bgSize = gpuSeg:GetDetectSize()
  local bgSizeInv = mf.vector2(1.0/bgSize:x(), 1.0/bgSize:y())

  local maskTex = gpuSeg:GetMask()
  local maskData = gpuSeg:GetData()

  if maskTex == nil or maskData == nil then
    return
  end

  local maskPos = maskData[dtPosition];
  local maskSize = maskData[dtSize];
  local maskPivot = maskData[dtPivot];
  local maskRoll = maskData[dtRoll];

  local maskMat =
    mf.Matrix33:CreateScaleMatrix(mf.vector2(bgSize:x(), bgSize:y())) *
    mf.Matrix33:CreateTranslateMatrix(mf.vector2(-maskPos[1], maskPos[2] - bgSize:y())) *
    mf.Matrix33:CreateTranslateMatrix(mf.vector2(-maskPivot[1], maskPivot[2])) *
    mf.Matrix33:CreateRotateMatrixZ(-maskRoll[1]) *
    mf.Matrix33:CreateTranslateMatrix(mf.vector2(maskPivot[1], -maskPivot[2])) *
    mf.Matrix33:CreateScaleMatrix(mf.vector2(1.0 / maskSize[1], -1.0 / maskSize[2]))

  postEffect:SetParameter(paramSet.stuff, true)
  postEffect:SetParameter(paramSet.mask, maskTex)
  postEffect:SetParameter(paramSet.maskMat, maskMat)
  
  local RV = face106:Get(refRadius[1] + 1) - face106:Get(refRadius[2] + 1)
  local R = RV:Length() * 0.5
  local A = maskSize[1] / maskSize[2]
  local A2 = A * A
  local refRectW = math.sqrt(A2 / (A2 + 1)) * R
  local refRectH = refRectW / A
  local refRects = {}
  for i = 1, 4 do
    local item = anchorsTable[i]
    local p1 = face106:Get(item[1] + 1)
    local p2 = face106:Get(item[2] + 1)
    local p = p1 * (1.0 - item[3]) + p2 * item[3]
    local refRect = mf.vector4(
      (p:x() - refRectW) * bgSizeInv:x(),
      (p:y() - refRectH) * bgSizeInv:y(),
      (refRectW * 2) * bgSizeInv:x(),
      (refRectH * 2) * bgSizeInv:y()
    )
    table.insert(refRects, refRect)
  end
  postEffect:SetParameter(paramSet.refRect1, refRects[1])
  postEffect:SetParameter(paramSet.refRect2, refRects[2])
  postEffect:SetParameter(paramSet.refRect3, refRects[3])
  postEffect:SetParameter(paramSet.refRect4, refRects[4])
  postEffect:SetParameter("bgSize", bgSize)
  --WARNING("SUNTYLOG: bgSize " .. string.format("%d %d", bgSize:x(), bgSize:y()))
end

function SkinStuffBehavior:_OnUpdate(delta)
  local camera = self.camera
  if camera == nil then
    WARNING("SkinStuffBehavior: no camera")
    return
  end
  local postEffect = camera:FindPostEffect("skin_stuff")
  if postEffect == nil then
    WARNING("SkinStuffBehavior: no skin_stuff post effect")
    return
  end
  local gpuSegs = self:_CollectGPUSeg()
  if #gpuSegs == 0 then
    WARNING("SkinStuffBehavior: no gpuSegs")
    return
  end
  --Diabled all skins first
  for _, ps in pairs(paramTable) do
    postEffect:SetParameter(ps.stuff, false)
  end
  for i = 1, #gpuSegs do
    if gpuSegs[i]:isActiveHierarchy() then
      self:_ProcessByGPUSeg(postEffect, gpuSegs[i])
    end
  end
end


SkinStuffBehavior:MemberRegister(
  "camera",
  vc.ScriptTypes.ReferenceType(ae.IContent:RTTI())
)


return SkinStuffBehavior