local venuscore = require "venuscore"
local mathfunction = require "mathfunction"
local apolloengine = require "apolloengine"
local libmasquerade = require "libmasque"

local Freezer = require "behavior.tracking_behavior.freezer"

local TRACKING_BS_ALL = {
  "browDown_L",
  "browDown_R",
  "browInnerUp",
  "browOuterUp_L",
  "browOuterUp_R",
  "cheekPuff",
  "eyeBlink_L",
  "eyeBlink_R",
  "eyeLookDown_L",
  "eyeLookDown_R",
  "eyeLookIn_L",
  "eyeLookIn_R",
  "eyeLookOut_L",
  "eyeLookOut_R",
  "eyeLookUp_L",
  "eyeLookUp_R",
  "eyeSquint_L",
  "eyeSquint_R",
  "eyeWide_L",
  "eyeWide_R",
  "jawForward",
  "jawLeft",
  "jawOpen",
  "jawOpenMouthClose",
  "jawRight",
  "mouthSmile_L",
  "mouthSmile_R",
  "mouthDimple_L",
  "mouthDimple_R",
  "mouthFrown_L",
  "mouthFrown_R",
  "mouthFunnel",
  "mouthLeft",
  "mouthLowerDown_L",
  "mouthLowerDown_R",
  "mouthPucker",
  "mouthRight",
  "mouthRollLower",
  "mouthRollUpper",
  "mouthShrugLower",
  "mouthShrugUpper",
  "mouthStretch_L",
  "mouthStretch_R",
  "mouthTightener",
  "mouthUpperUp_L",
  "mouthUpperUp_R",
  "noseSneer_L",
  "noseSneer_R"
}

local CONFLICT_BS = {
  {"mouthFunnel", "mouthPucker"}
}

local BS_NONE = "None"

local FaceExpression = venuscore.VenusBehavior:extend("FaceExpressionBehavior");

function FaceExpression:new()
  self.expression_freezer = Freezer()
end

function FaceExpression:Freeze(active)
  self.expression_freezer:Freeze(active)
end

function FaceExpression:_Play()
end

function FaceExpression:_OnAwake()
  if self.MorpherNode == nil then
    self.MorpherNode = self.Node
    self:OnMorpherNodeChanged(self.MorpherNode)
  else
    self.MorpherComp = self.MorpherNode:GetComponent(apolloengine.Node.CT_MORPH)
    if self.MorpherGroup == nil then
      self.MorpherGroup = self.MorpherComp:GetGroupName(0)
    end
    self:CollectMorpherInfo()
    --self:MatchBsNames()
  end

  if self.Tracking == nil then
    self.Tracking = self.Node:GetComponent(apolloengine.Node.CT_TRACK)
  end
end

function FaceExpression:OnMorpherNodeChanged(node)
  if node ~= nil then
    self.MorpherNode = node
    self.MorpherComp = node:GetComponent(apolloengine.Node.CT_MORPH)
    self.MorpherGroup = self.MorpherComp:GetGroupName(0)
    self:CollectMorpherInfo()
    self:MatchBsNames()
  end
end

function FaceExpression:OnMorpherGroupChanged(group)
  self.MorpherGroup = group
  self:CollectMorpherInfo()
  self:MatchBsNames()
end

function FaceExpression:MatchBsNames()
  local original_morpherbs_names, morpherbs_identifiers = self:GetMorpherBsNames()
  local original_trackerbs_names, trackerbs_identifiers = self:GetTrackingBsNames()
  
  local thiz = self

  for i = 1, #TRACKING_BS_ALL do
     thiz[TRACKING_BS_ALL[i]] = BS_NONE
  end

  for i = 1, #morpherbs_identifiers do
    local morpher_name = morpherbs_identifiers[i]
    local bs_len = #morpher_name
    for j = 1, #trackerbs_identifiers do   
      local bs_name = trackerbs_identifiers[j]
      if thiz[original_trackerbs_names[j]] == BS_NONE and morpher_name:find(bs_name, -bs_len, true) then
         thiz[original_trackerbs_names[j]] = original_morpherbs_names[i]
         goto CONTINUE1
      end    
    end
    ::CONTINUE1::
  end
end

function FaceExpression:_OnStart()
  
end

function FaceExpression:_OnUpdate(timespan)
    local thiz = self
    if self.Tracking and self.MorpherInfos and self.Tracking:IsTracked() then
      local weight_dict = self:GetTrackedBsWeight()
      weight_dict = self.expression_freezer:Process(weight_dict)
      for trackerBSName, w in pairs(weight_dict)
      do 
        if thiz[trackerBSName] ~= nil then
          local morpherBSName = thiz[trackerBSName]
          if self.MorpherInfos[morpherBSName] ~= nil then
             self.MorpherInfos[morpherBSName] = {weight = w}
 if not _KRATOSEDITOR then
            self.MorpherComp:SetChannelPercents(self.MorpherGroupIndex, self.MorpherInfos[morpherBSName]["id"], w)
 end
          end
        end
      end

      self.MorpherComp:Update(timespan)      
      self.expression_freezer:Update()
    end
end

function FaceExpression:GetMorpherNode()
    return self.MorpherNode;
end

function FaceExpression:SetMorpherNode(node)
    self:OnMorpherNodeChanged(node)
end

function FaceExpression:GetMorpherGroup()
    return self.MorpherGroup;
end

function FaceExpression:SetMorpherGroup(group)
  if self.MorpherGroup == nil then
    self.MorpherGroup = group
  else
    self:OnMorpherGroupChanged(group)
  end
end

function FaceExpression:CollectMorpherInfo()
  self.MorpherInfos = {}
  if self.MorpherComp ~= nil and self.MorpherGroup ~= nil then
    self.MorpherGroupIndex = self.MorpherComp:GetGroupIndexByName(self.MorpherGroup)
    for i = 1, self.MorpherComp:GetNumberOfChannel(self.MorpherGroupIndex) 
    do
      local channel_index = i - 1
      local cname = self.MorpherComp:GetChannelName(self.MorpherGroupIndex, channel_index)
      self.MorpherInfos[cname] = { id = channel_index }
    end
  end
end

function FaceExpression:GetMorpherBsNames()
  local bs_names = {}
  local bs_identifiers = {}
  
  if self.MorpherInfos ~= nil then
    for bs_name, _ in pairs(self.MorpherInfos)
    do
      local lower_bs_name = bs_name:lower()
      
      table.insert(bs_names, bs_name)
      table.insert(bs_identifiers, lower_bs_name)
    end
  end
  return bs_names, bs_identifiers
end

function FaceExpression:GetTrackingBsNames()
  local bs_names = {}
  local bs_identifiers = {}
  
  for i = 1, #TRACKING_BS_ALL 
  do
    local bs_name = TRACKING_BS_ALL[i]
    local lower_bs_name = bs_name:lower()

    table.insert(bs_names, bs_name)
    table.insert(bs_identifiers, lower_bs_name)
  end
  return bs_names, bs_identifiers
end

function FaceExpression:GetTrackedBsWeight()
  local num = self.Tracking:GetBlendshapeNumber()
  local weight_dict = {}
  for i = 1, num do 
    local weight = self.Tracking:GetBlendshapeWeight(i - 1)
    local name = self.Tracking:GetBlendshapeName(i - 1)
    if name == "mouthFunnel" or name == "mouthPicker" then
      weight_dict[name] = weight * 0.6
    elseif name == "mouthRollUpper" or name == "mouthRollLower" then
      weight_dict[name] = weight * 1.0
    else
      weight_dict[name] = weight * 0.75
    end
  end
  --weight_dict = self:ProcessConflicts(weight_dict)
  --weight_dict["mouthFunnel"] = weight_dict["mouthFunnel"] * 0.1
  return weight_dict
end

function FaceExpression:ProcessConflicts(weight_dict)
  for _, conflicts in ipairs(CONFLICT_BS) do
    local max_bs = 0
    local max_weight = -1
    for i = 1, #conflicts do
      local bs = conflicts[i]
      if weight_dict[bs] ~= nil and weight_dict[bs] > max_weight then
          max_bs, max_weight = i, weight_dict[bs]
      end
    end
    for i = 1, #conflicts do
      if i ~= max_bs then
        local bs = conflicts[i]
        if weight_dict[bs] ~= nil then
          weight_dict[bs] = weight_dict[bs] * 0.2
        end 
      end
    end
  end
  return weight_dict
end

function FaceExpression:GetMorphGroup()
  local morph_groups = {}
  if self.MorpherComp ~= nil then
    for i = 1, self.MorpherComp:GetNumberOfGroup() 
    do
      table.insert(morph_groups, self.MorpherComp:GetGroupName(i-1))
    end
  end
  return morph_groups
end

FaceExpression:MemberRegister("Tracking",
    venuscore.ScriptTypes.ReferenceType(
        libmasquerade.TrackingComponent:RTTI()
));

FaceExpression:MemberRegister("MorpherNode",
    venuscore.ScriptTypes.ReferenceType(
        apolloengine.Node:RTTI(),
        FaceExpression.GetMorpherNode,
        FaceExpression.SetMorpherNode
));

FaceExpression:MemberRegister("MorpherGroup",
    venuscore.ScriptTypes.ComboType(
     function(thiz)
       local result = {}
       local group_names = thiz:GetMorphGroup()
       for _, name in ipairs(group_names) do
            table.insert(
                result,
                {
                    key = name,
                    value = name
                }
            )
        end          
        return result
      end,
      FaceExpression.GetMorpherGroup,
      FaceExpression.SetMorpherGroup
  )); 


for i=1, #TRACKING_BS_ALL do
    FaceExpression:MemberRegister(TRACKING_BS_ALL[i],
    venuscore.ScriptTypes.ComboType(
     function(thiz)
       local result = {}
         table.insert(
           result,
           {
              key = BS_NONE,
              value = BS_NONE
           }
         )
         local bs_names = thiz:GetMorpherBsNames()
         for _, name in ipairs(bs_names) do
                  table.insert(
                      result,
                      {
                          key = name,
                          value = name
                      }
                  )
          end          
          return result
      end
  )); 
end

return FaceExpression;