local Object        = require "classic"
local venuscore     = require "venuscore"
local venusjson     = require "venusjson"
local apolloengine  = require "apollocore"
local mathfunction  = require "mathfunction"
local utils         = require "behavior.cartoon_behavior.utils"

local EyeDeformation = venuscore.VenusBehavior:extend("EyeDeformationBehavior")

function EyeDeformation:new()
  EyeDeformation.super.new(self)
  self.Config = ""
end

function EyeDeformation:Clear()
  self.mesh_deformation = nil
end

function EyeDeformation:_OnAwake()
  if self.Config ~= "" then
    self:LoadConfig()
  end
  if (self.FaceNode ~= nil) then
    local faceTransfer = utils:GetBehavior(self.FaceNode, "facetransfer")
    if faceTransfer then
      faceTransfer:RegisterFaceTransferedEvent(self, self._OnTransferFinished)
      faceTransfer:RegisterFaceTransferStartEvent(self, self._OnTransferStart)
    end
  end  
end

function EyeDeformation:_OnTransferStart(vertexData, channelVertexData)
  if vertexData == nil or channelVertexData == nil then
    return
  end
  
  if self.scale_boundary ~= nil then
    self.origin_boundary = {}
    for _, point_pair in ipairs(self.scale_boundary) do
      table.insert(self.origin_boundary, {
          mathfunction.vector3(unpack(vertexData[point_pair[1]+1])),
          mathfunction.vector3(unpack(vertexData[point_pair[2]+1]))})
    end
    if self.position_bind then
      local bind_center_index = self.position_bind.center.bind
      local render = self.Node:GetComponent(apolloengine.Node.CT_RENDER)
      local mesh_stream = render:GetVertexStream()
      local mesh_vtx = mesh_stream:GetVertexData()
      self.bind_center = mathfunction.vector3(unpack(mesh_vtx[bind_center_index+1]))  
      
      local bind_transfered_to = self:GetCenterBarycentric(vertexData, channelVertexData)
      self.origin_center = bind_transfered_to - self.bind_center
    end
  end
end


function EyeDeformation:_OnTransferFinished(vertexData, channelVertexData)
  if vertexData == nil or channelVertexData == nil then
    return
  end
  
  local transform = self.Node:GetComponent(apolloengine.Node.CT_TRANSFORM)
  local scale = transform:GetLocalScale()

  if self.scale_boundary ~= nil then
    local max_scale = 0
    for i = 1, #self.scale_boundary do      
      local transfered1 = mathfunction.vector3(unpack(vertexData[self.scale_boundary[i][1]+1]))
      local transfered2 = mathfunction.vector3(unpack(vertexData[self.scale_boundary[i][2]+1]))
      local cur_scale = (transfered1-transfered2):Length() / (self.origin_boundary[i][1]-self.origin_boundary[i][2]):Length()
      if cur_scale > max_scale then
        max_scale = cur_scale
      end
    end
    scale = scale * max_scale
    if max_scale ~= 1 then
      transform:SetLocalScale(scale)
    end
  end
  
  if self.position_bind ~= nil then    
    local bind_transfered_to = self:GetCenterBarycentric(vertexData, channelVertexData)
    if bind_transfered_to == nil then
      bind_transfered_to = self.bind_center
    end
    
    local radius_vec = self.bind_center * scale
    local transfered_center = bind_transfered_to - radius_vec
    local radius2 = radius_vec:LengthPow()
    
    local max_t = 0
    local mesh = self.position_bind.mesh
    for _, bs in ipairs(mesh.blendshape) do
      for _, vertex in ipairs(mesh.vertex) do
        if channelVertexData[bs] then
          local index = vertex+1
          local target = mathfunction.vector3(unpack(channelVertexData[bs][index])) + mathfunction.vector3(unpack(vertexData[index]))
          local start_target = target - transfered_center
          local a = radius2
          local b = 2 * start_target:Dot(radius_vec)
          local c = start_target:LengthPow() - radius2
          if c < 0 then
            local t = (-b + math.sqrt(b * b - 4 * a * c)) / 2 / a
            if t > max_t then
              max_t = t
            end
          end
        end
      end
    end
    transfered_center = transfered_center - radius_vec * max_t
    local translate = transfered_center - self.origin_center
    transform:SetLocalPosition(transform:GetLocalPosition() + translate)
  end
end


function EyeDeformation:_OnUpdate(delta)

end

function EyeDeformation:LoadConfig()
  local rootconfig = venusjson.LaodJsonFile(self.Config)
  self.scale_boundary = rootconfig.scale_boundary
  self.position_bind = rootconfig.position_bind
end

function EyeDeformation:SetConfig(value)
  self.Config = value
  self:LoadConfig()
end

function EyeDeformation:GetConfig()
  return self.Config
end

function EyeDeformation:GetCenterBarycentric(vertexData, channelVertexData)
  local center_bs = self.position_bind.center.blendshape
  local bind_transfered_to = mathfunction.vector3(0.0,0.0,0.0)
  local center_to = self.position_bind.center.to
  if channelVertexData[center_bs] then
    for i = 1,#center_to.triangle do
      local index = center_to.triangle[i]+1
      bind_transfered_to
        = bind_transfered_to
        + (mathfunction.vector3(unpack(channelVertexData[center_bs][index])) + mathfunction.vector3(unpack(vertexData[index])))
        * center_to.barycentric[i]
    end
    return bind_transfered_to
  end
  LOG("cannot find blendshape "..center_bs.." in facetransfer")
  return nil
end

EyeDeformation:MemberRegister("Config",  
  venuscore.ScriptTypes.FilePathType(
    {"json"},
    EyeDeformation.GetConfig,
    EyeDeformation.SetConfig))

EyeDeformation:MemberRegister("FaceNode",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI()))

return EyeDeformation