#define THREAD_GROUP_SIZE 256
#define THREAD_SIZE 8

float4 _Size;
float3 _gridRangeMax;
float3 _gridRangeMin;

float3 _gridRange;

float  _TimeStep;
float  _Dissipation;

float  _Forward;

inline uint GridCord2GlobalIndex(uint ix, uint iy)
{
	ix = max(ix, (uint)0);
  iy = max(iy, (uint)0);
	ix = min(ix, _Size.x - 1);
  iy = min(iy, _Size.y - 1);
  return iy * _Size.x + ix;
}

inline uint GridCord2GlobalIndex(uint ix, uint iy, uint iz)
{
	ix = max(ix, (uint)0);
  iy = max(iy, (uint)0);
	iz = max(iz, (uint)0);
	ix = min(ix, _Size.x - 1);
  iy = min(iy, _Size.y - 1);
	iz = min(iz, _Size.z - 1);
  return iz * (_Size.x * _Size.y) + iy * _Size.x + ix;
}

inline uint GridCord2GlobalIndex(uint3 idx)
{
	idx.x = max(idx.x, (uint)0);
  idx.y = max(idx.y, (uint)0);
	idx.z = max(idx.z, (uint)0);
	idx.x = min(idx.x, _Size.x - 1);
  idx.y = min(idx.y, _Size.y - 1);
	idx.z = min(idx.z, _Size.z - 1);
  return idx.z * (_Size.x * _Size.y) + idx.y * _Size.x + idx.x;
}

inline float3 GridCord2WorldPos(float3 coord)
{
	return coord * (_gridRangeMax - _gridRangeMin) / (_Size.xyz - float3(1.0, 1.0, 1.0)) + _gridRangeMin;
}

inline float3 WorldPos2GridCord(float3 pos)
{
	pos = clamp(pos, _gridRangeMin, _gridRangeMax) - _gridRangeMin;
	return pos * (_Size.xyz - float3(1.0, 1.0, 1.0)) / (_gridRangeMax - _gridRangeMin);	
}

inline float WorldPosX2GridCordX(float pos)
{
	pos = clamp(pos, _gridRangeMin.x, _gridRangeMax.x) - _gridRangeMin.x;
	return pos * (_Size.x - 1.0) / (_gridRangeMax.x - _gridRangeMin.x);
}

inline float WorldPosY2GridCordY(float pos)
{
	pos = clamp(pos, _gridRangeMin.y, _gridRangeMax.y) - _gridRangeMin.y;
	return pos * (_Size.y - 1.0) / (_gridRangeMax.y - _gridRangeMin.y);
}

inline float WorldPosZ2GridCordZ(float pos)
{
	pos = clamp(pos, _gridRangeMin.z, _gridRangeMax.z) - _gridRangeMin.z;
	return pos * (_Size.z - 1.0) / (_gridRangeMax.z - _gridRangeMin.z);
}

inline float2 WorldPos2GridCord(float2 pos)
{
	pos = clamp(pos, -_gridRange.xy, _gridRange.xy) + _gridRange.xy;
	
	float scaleX = _Size.x / _gridRange.x / 2;
	float scaleY = _Size.y / _gridRange.y / 2;
	
	float _x = min(pos.x * scaleX, _Size.x - 1);
	float _y = min(pos.y * scaleY, _Size.y - 1);
	
	return float2(_x, _y);
}


/* ------------------------------------- */
/* 3D sdf shape and normal function */

float sdBox(float3 p, float3 b )
{
  float3 q = abs(p) - b;
  return length(max(q,0.0)) + min(max(q.x,max(q.y,q.z)),0.0);
}

float3 calcBoxNormal(float3 pos, float3 b)
{
  float2 e = float2(1.0, -1.0) * 0.5773 * 0.0005;
  return normalize(e.xyy * sdBox(pos + e.xyy, b).x + e.yyx * sdBox(pos + e.yyx, b).x +
					  e.yxy * sdBox(pos + e.yxy, b).x + e.xxx * sdBox(pos + e.xxx, b).x);
}

float sdCappedCylinder( float3 p, float h, float r )
{
  float2 d = abs(float2(length(p.xz), p.y)) - float2(r, h);
  return min(max(d.x, d.y),0.0) + length(max(d, 0.0));
}

float3 calcCylinderNormal(float3 pos, float h, float r)
{
  float2 e = float2(1.0, -1.0) * 0.5773 * 0.0005;
  return normalize(e.xyy * sdCappedCylinder(pos + e.xyy, h, r).x + e.yyx * sdCappedCylinder(pos + e.yyx, h, r).x +
					  e.yxy * sdCappedCylinder(pos + e.yxy, h, r).x + e.xxx * sdCappedCylinder(pos + e.xxx, h, r).x);
}

float sdVerticalCapsule( float3 p, float h, float r )
{
  p.y -= clamp( p.y, 0.0, h );
  return length( p ) - r;
}

float3 calcCapsuleNormal(float3 pos, float h, float r)
{
  float2 e = float2(1.0, -1.0) * 0.5773 * 0.0005;
  return normalize(e.xyy * sdVerticalCapsule(pos + e.xyy, h, r).x + e.yyx * sdVerticalCapsule(pos + e.yyx, h, r).x +
					  e.yxy * sdVerticalCapsule(pos + e.yxy, h, r).x + e.xxx * sdVerticalCapsule(pos + e.xxx, h, r).x);
}

float sdSphere( float3 p, float s )
{
  return length(p)-s;
}

float3 calcSphereNormal(float3 pos, float b)
{
  float2 e = float2(1.0, -1.0) * 0.5773 * 0.0005;
  return normalize(e.xyy * sdSphere(pos + e.xyy, b).x + e.yyx * sdSphere(pos + e.yyx, b).x +
					  e.yxy * sdSphere(pos + e.yxy, b).x + e.xxx * sdSphere(pos + e.xxx, b).x);
}

float sdConeHP(float3 p, float2 c, float h )
{
  // c is the sin/cos of the angle, h is height
  // Alternatively pass q instead of (c,h),
  // which is the point at the base in 2D
  float2 q = h * float2(c.x / c.y, -1.0);
    
  float2 w = float2( length(p.xz), p.y );
  float2 a = w - q * clamp( dot(w,q)/dot(q,q), 0.0, 1.0 );
  float2 b = w - q * float2( clamp( w.x/q.x, 0.0, 1.0 ), 1.0 );
  float k = sign( q.y );
  float d = min(dot( a, a ),dot(b, b));
  float s = max( k * (w.x * q.y - w.y * q.x),k * (w.y - q.y)  );
  return sqrt(d) * sign(s);
}

float3 calcConeNormal(float3 pos, float2 c, float h )
{
  float2 e = float2(1.0, -1.0) * 0.5773 * 0.0005;
  return normalize(e.xyy * sdConeHP(pos + e.xyy, c, h).x + e.yyx * sdConeHP(pos + e.yyx, c, h).x +
					  e.yxy * sdConeHP(pos + e.yxy, c, h).x + e.xxx * sdConeHP(pos + e.xxx, c, h).x);
}

/* sd shape and caculate normal function */
/* ------------------------------------- */

float3 opRotation3D(float3 pt, float angle) {
    
    // Invert angle
    float cosineAngle = cos(-angle);
    float sineAngle = sin(-angle);
    
    // Generate rotation matrix
    float3x3 rotation = float3x3(
        float3(cosineAngle, 0.0, -sineAngle),
        float3(0.0,         1.0, 0.0),
        float3(sineAngle,   0.0, cosineAngle)
    );
    
    // Apply rotation
    return mul(rotation, pt);
}

float sdSphere( float2 p, float s )
{
  return length(p)-s;
}

float sdfBox(float2 p, float2 size)
{
	float2 d = abs(p) - size;
	return length(max(d, float2(0, 0))) + min(max(d.x, d.y), 0.0);
}

float sdConeLP( float3 p, float2 c, float h )
{
  float q = length(p.xz);
  return max(dot(c.xy, float2(q, p.y)), -h - p.y);
} 

float sdCapsule( float3 p, float3 a, float3 b, float r )
{
  float3 pa = p - a, ba = b - a;
  float h = clamp( dot(pa, ba) / dot(ba, ba), 0.0, 1.0 );
  return length( pa - ba * h ) - r;
}

float sdOrientedBox(float2 pos, float2 a, float2 size, float thickness)
{
	float l = length(size - a);

	float2  d = (size - a) / l;

	float2  q = (pos - (a + size) * 0.5);
	q = mul(float2x2(d.x, -d.y, d.y, d.x), q);
	q = abs(q) - float2(l, thickness) * 0.5;

	return length(max(q, 0.0)) + min(max(q.x, q.y), 0.0);
}

float sdudRoundBox(float2 p, float2 b, float r)
{
	return length(max(abs(p) - b, 0.0)) - r;
}

float2 opRotation(float2 pt, float angle)
{
	// Invert angle
	float cosineAngle = cos(-angle);
	float sineAngle = sin(-angle);

	// Generate rotation matrix
	float2x2 rotation = float2x2(
		float2(cosineAngle, -sineAngle),
		float2(sineAngle, cosineAngle)
	);

	// Apply rotation
	return (mul(rotation, pt));
}