#DEFPARAMS
#END

#DEFTAG
ShaderName = "Mouth"
RenderQueue = "Opaque"
#END

#DEFPASS Always
COLOR_MASK   = COLOR_RGBA
ALPAH_MODE   = { ALPAH_OFF }
DRAW_MODE    = { CULL_FACE_BACK, DEPTH_MASK_ON, DEPTH_TEST_ON, DEPTH_FUNCTION_LESS }
STENCIL_MODE = {STENCIL_OFF}
LIGHT_MODE = { ALWAYS }

CGPROGRAM
#pragma vertex vert
#pragma fragment frag

#include "functions/liveportrait.inc"
#include "common.inc"

struct appdata   
{
    float4 vertex           : POSITION;
    float2 uv               : TEXCOORD0;
    float3 normal           : NORMAL;
    float4 tangent          : TANGENT;
};

struct v2f
{
    float2 uv               : TEXCOORD0;
    float3 posInCam         : TEXCOORD1;
    float3 T                : TEXCOORD2;
    float3 B                : TEXCOORD3;
    float3 N                : TEXCOORD4;
    float  clippedZ         : COLOR;
    float3 normal           : NORMAL;
    float4 vertex           : SV_POSITION;
};

float4x4 PROJECTION;
float4x4 MODELVIEW;
float4x4 INVMODELVIEW;

Texture2D TEXTURE_DIFFUSE;
Texture2D TEXTURE_BUMP;
SamplerState TEXTURE_DIFFUSE_Sampler;
SamplerState TEXTURE_BUMP_Sampler;

float     ENABLETEXTURE;
float     ENABLENORMAL;

float3    AMBIENT;
float3    SPECULAR;
float3    EMISSION;
float3    DIFFUSE;
float     SHININESS;
float4    LIGHTPOSN;
float4    LIGHTCOLOR;

v2f vert(appdata v)
{
    v2f o;
    
    float4 vtxPositionInCam = mul(v.vertex, MODELVIEW);
    o.vertex = UniformNDC(mul(vtxPositionInCam, PROJECTION));

    o.posInCam = vtxPositionInCam.xyz;
    o.normal = normalize(mul(float4(v.normal, 0.0), INVMODELVIEW).xyz);
    o.uv = v.uv;

    // construct TBN matrix for normal map
    float3 T = normalize(mul(float4(v.tangent.xyz, 0.0), INVMODELVIEW).xyz);
    float3 N = normalize(mul(float4(v.normal, 0.0), INVMODELVIEW).xyz);
    
    // Gauss-Schmidt orthogonalization
    T = normalize(T - dot(T, N) * N);
    float3 B = cross(N, T) * v.tangent.w;
    o.T = T;
    o.B = B;
    o.N = N;

    // most mouth depth will fall in range [min_z, max_z]
    float min_z = 0.05;
    float max_z = 0.25;
    float clamped_z = clamp(o.vertex.z / o.vertex.w, min_z, max_z);

    o.clippedZ = (clamped_z - min_z) / (max_z - min_z);
    
    return o;
}

void frag(in v2f i, out float4 mainColor : SV_Target0)
{
    float3 lightColor = AMBIENT + EMISSION;

    // eye direction
    float3 eyeDir = normalize(-i.posInCam);
    
    float3x3 TBN = float3x3(i.T, i.B, i.N);

    float3 normalDir;
    if (ENABLENORMAL > 0.5)
    {
        float3 normal_sample = TEXTURE_BUMP.Sample(TEXTURE_BUMP_Sampler, i.uv).rgb;
        normal_sample = normalize(normal_sample * 2.0 - 1.0);
        normal_sample = normalize(mul(normal_sample, TBN));
        normalDir = normal_sample;
    }
    else
    {
        normalDir = i.normal;
    }

    // light direction
    float3 lightDir;
    if(0.0 == LIGHTPOSN.w)	// directional light
    {
        lightDir = normalize(LIGHTPOSN.xyz);
    }
    else
    {
        float3 lightPos = LIGHTPOSN.xyz / LIGHTPOSN.w;
        lightDir = normalize(lightPos - i.posInCam);
    }

    float3 halfVec = normalize(eyeDir + lightDir);
    float3 color = ComputeLight(LIGHTCOLOR.rgb, normalDir, lightDir, halfVec, DIFFUSE, SPECULAR, SHININESS);
    float depthLightIntensity = clamp(-log(i.clippedZ), 0.0, 1.0);
    lightColor += depthLightIntensity * color * LIGHTCOLOR.a;

    float4 finalColor;
    if (ENABLETEXTURE > 0.5)
    {
        finalColor = float4(lightColor * TEXTURE_DIFFUSE.Sample(TEXTURE_DIFFUSE_Sampler, i.uv).rgb, 1.0);
    }
    else
    {
        finalColor = float4(lightColor, 1.0);
    }
    mainColor = finalColor;
}
ENDCG
#END



