local vc = require "libvenuscore"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local ArgInfo = require "bluecore.core.arg_info"

local DispatcherInfo = vc.Object:extend("DispatcherInfo");

function DispatcherInfo:new(classInfo)
  self.owner = classInfo ;
  self._RefactorList = setmetatable({}, { __mode = 'v' })
end

--  如果是直接注册的blueprint 代表是蓝图类 而不是prefab类 或者是端内运行的情况
-- register from bluefunction instance
function DispatcherInfo:RegisterEventDispatcher(dispatcher)

  self._inputInfos = {}

  self._myName = dispatcher:GetName();
  self._myUid = dispatcher:GetUid();

  local inputInfos = dispatcher:GetInputInfo();

  -- input argument
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterBlueFunPinInfo(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end

  dispatcher:AddRefactor(self); -- 只是监听其中一个场景的
end



function DispatcherInfo:RegisterPrefab(prefabJson)

  self._inputInfos = {}

  -- dispatcher name & uid
  self._myName = prefabJson.name ;
  self._myUid =  prefabJson.uid ;

  local inputInfos = prefabJson.inputInfos;

  -- input argument
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterPrefab(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end
end

function DispatcherInfo:GetInputInfo()
  return  self._inputInfos ;
end



function DispatcherInfo:GetName()
  return self._myName ;
end

function DispatcherInfo:GetUid()
  return self._myUid ;
end



function DispatcherInfo:AddRefactor(iRefactorCallback)

  if _KRATOSEDITOR then

    local cb = iRefactorCallback ;

    if (cb.EditorPinAdd == nil) then error("EditorPinAdd not implement") end
    if (cb.EditorPinDelete == nil) then error("EditorPinDelete not implement") end
    if (cb.InfoChange == nil) then error("InfoChange not implement") end
    if (cb.NameChanged == nil) then error("NameChanged not implement") end

    if (cb.ToBeDeletedEd == nil) then error("ToBeDeletedEd not implement") end -- ToBeDeletedEd 是类信息删除
    if (cb._OnDeleteEd == nil) then error("_OnDeleteEd not implement"); end    -- DeleteEd _OnDeleteEd 蓝图节点删除

    if (cb._OnDeserializePost == nil) then error("_OnDeserializePost not implement") end   -- 恢复引脚类型依赖的一些类信息(因为类在_OnDeserialize阶段才注册)
    if (cb._OnCreateWithEditor == nil) then error("_OnCreateWithEditor not implement") end -- 编辑器创建或者加载素材需要加入 引用列表

    table.insert(self._RefactorList, cb);

  end -- 非编辑器启动不加入refactor列表

end


function DispatcherInfo:InfoChange(reflectIdx, type, val, isInput)
  --  更新自己的信息

  local argInfo = isInput and self._inputInfos or self._outputInfos
  argInfo = argInfo[reflectIdx]

  if type == BD.FuncInfoProp.Name then
    argInfo:ChangeName(val)
  elseif type == BD.FuncInfoProp.BaseType then
    argInfo:ChangeRtti(val, BU:GetDefaultByRtti(val));
  elseif type == BD.FuncInfoProp.Value then
    argInfo:ChangeDefault(val);
  end

  -- 通知引用该函数的节点
  for _, cb in pairs(self._RefactorList) do
    cb:InfoChange(reflectIdx, type, val, isInput);
  end

end

function DispatcherInfo:EditorPinDelete(isInput, idx)

  if isInput then
    -- _inputInfos是数组 移除前面的后面自动移前
    table.remove(self._inputInfos, idx)
  else
    table.remove(self._outputInfos, idx)
  end

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinDelete(isInput, idx);
  end

end


function DispatcherInfo:EditorPinAdd(isInput, reflectInfo)

  local argInfo = ArgInfo()
  argInfo:RegisterBlueFunPinInfo(reflectInfo);
  if isInput then
    table.insert(self._inputInfos,  argInfo);
  else
    table.insert(self._outputInfos, argInfo);
  end

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinAdd(isInput, reflectInfo);
  end

end


function DispatcherInfo:RemoveRefactor(callback)
  if _KRATOSEDITOR then
    for idx, cb in pairs(self._RefactorList) do
      if cb == callback then
        self._RefactorList[idx] = nil
      end
    end
  end
end

function DispatcherInfo:ToBeDeletedEd()
  for _, cb in pairs(self._RefactorList) do
    cb:ToBeDeletedEd();
  end
  self.owner:ToBeDeletedEd(self)
  self.owner = nil
end

function DispatcherInfo:NameChanged(oldName, newName)
  self._myName = newName
  for _, cb in pairs(self._RefactorList) do
    cb:NameChanged(oldName, newName);
  end
end

function DispatcherInfo:Dump()
  LOG("----name:"..tostring(self._myName))
  LOG("---- uid:"..tostring(self._myUid))
  for i, argInfo in pairs( self._inputInfos) do
    LOG("---- arg "..tostring(i)..":")
    argInfo:Dump();
  end
end

function DispatcherInfo:GetRefactorList()
  return self._RefactorList;
end

return DispatcherInfo