--[[
页面撤销逻辑说明：
1、从编辑页进入touch magic页面时，先保存上次添加的touch magic，id存在storeNodesId
2、在touch magic页面有删除到storeNodesId里面的魔法时，先不从场景移除node，而是设为deactive状态，
   touchMagicNodes、nodeBehaviorInstances、touchCacheBehaviorInstances对应的待删除的数据，先保存一份到
   storeTouchMagicNodes、storeNodeBehaviorInstances、storeTouchCacheBehaviorInstances，再从原table删除
3、从touch magic页面返回编辑页时，
A. 撤销返回，touchMagicNodes、nodeBehaviorInstances、touchCacheBehaviorInstances删除后面新增的魔法，
   从storeTouchMagicNodes、storeNodeBehaviorInstances、storeTouchCacheBehaviorInstances恢复之前的魔法
B. 应用返回，把touchMagicNodes保存的节点彻底从场景删除，清空storeNodesId、storeTouchMagicNodes、storeNodeBehaviorInstances、storeTouchCacheBehaviorInstances

提前加载素材：
1、首先给素材节点定义状态：0 none; 1 loaded; 2 applied; 3 previewed。none没有结点，1表示素材刚加载，2素材应用过，3素材预览过
2、touch每次应用完，自动加载同样的素材，加载之后素材结点状态是loaded
3、切换素材时，如有预加载的素材(loaded状态)，先删掉，再加载新的素材
4、素材预览结束之后，reset之后状态变为loaded，可用于继续预览或应用
5、退出touch页面时，删除预加载结点
]]

local BundleSystem = require "venuscore.bundle.bundlesystem"
local mathfunction = require "mathfunction"
local apolloengine = require "apollocore"
local venuscore = require "venuscore"
local apollocore = require "apollocore"

local EditorTouchMagicManager = {}

local SCRIPT_PATH = "scrs:behavior/touch_cache_behavior.lua";

function EditorTouchMagicManager:Initialize(sequenceBase)
  self.resourceType = sequenceBase
  self.sequenceBase = sequenceBase
  self.touchMagicRootNode = nil;
  self.cameraRenderTarget = nil;
  self.touchMagicNodes = {};
  self.nodeBehaviorInstances = {};
  self.touchCacheBehaviorInstances = {};
  
  --页面状态保存和撤销恢复
  self.storeNodesId = {}; --node id列表
  self.storeTouchMagicNodes = {};
  self.storeNodeBehaviorInstances = {};
  self.storeTouchCacheBehaviorInstances = {};
  
  self.lastLoadedPath = nil; -- 记录最新加载的素材的路径，如果该素材被删，置nil。主要用于保存预加载素材的路径
  self.curNodeState = 0; -- 0 none; 1 loaded; 2 applied; 3 previewed
  
  self.touchCacheCount = 0;

  self.lastTouchPoint = nil;                --记录开始应用前的触屏位置
  self.needStartApply = false;              --收到有效的触屏信息后，是否开始应用
end

function EditorTouchMagicManager:LoadTouchMagic(cutterScene, path, rootconfig)
  ERROR("EditorTouchMagicManager:LoadTouchMagic " .. path)
  
  --如果前面有预加载的结点，先删除
  if self.curNodeState == 1 then
    local res = self:_UnloadLastTouchMagic(cutterScene, true, true);
    if not res then
      ERROR("LoadTouchMagic remove preload node fail, count=%d", #self.touchMagicNodes);
      return false;
    end
    LOG("LoadTouchMagic remove preload node");
  end
  
  self:_CreateTouchRootNodeIfNeed(cutterScene)
  local touchNode = self:_CreateTouchNode(cutterScene, path, rootconfig)
  if touchNode ~= nil then
    self:_ConfigRenderTarget(touchNode, self.cameraRenderTarget)
    self.touchMagicRootNode:AttachNode(touchNode);
    table.insert(self.touchMagicNodes, touchNode);
    
    self:_SetParticleVisibility(#self.touchCacheBehaviorInstances, false);
    
    self.lastLoadedPath = path;
    self.curNodeState = 1;

    return true;
  end
  return false;
end

--@param removePreNode true表示无论是不是预加载的结点，都会被删除；false 会跳过预加载结点
function EditorTouchMagicManager:_UnloadLastTouchMagic(scene, removeAbsolutely, removePreNode)
  if scene == nil then
    LOG("_UnloadLastTouchMagic Error: scene is nil!");
    return false;
  end
  
  local delNodeIsPreload = (removePreNode == true and self.curNodeState == 1); --待删除的结点是预加载结点
  
  --如果有预加载的结点，要考虑跳过
  local skipNode = 0;
  if removePreNode ~= true and self.curNodeState == 1 then
    skipNode = 1;
  end
  local id = #self.touchMagicNodes - skipNode; --待删除结点的下标
  
  LOG("_UnloadLastTouchMagic id "..tostring(id));
  if(id > 0) then
    local touchNode = self.touchMagicNodes[id];
    if touchNode then
      if removeAbsolutely ~= true and self:_IsStoreNode(touchNode) then
        --上次添加的touch magic，先不真正删除
        touchNode.Active = false;
        table.insert(self.storeTouchMagicNodes, self.touchMagicNodes[id]);
        table.insert(self.storeTouchCacheBehaviorInstances, self.touchCacheBehaviorInstances[id]);
        table.insert(self.storeNodeBehaviorInstances, self.nodeBehaviorInstances[id]);
      else
        self.touchMagicRootNode:DetachNode(touchNode);
        scene:DeleteNode(touchNode);
      end 
      table.remove(self.touchMagicNodes, id);
      table.remove(self.touchCacheBehaviorInstances, id);
      table.remove(self.nodeBehaviorInstances, id);
      
      --待删除结点是预加载结点时，更新结点状态
      if delNodeIsPreload then
        self.lastLoadedPath = nil;
        if #self.touchMagicNodes > 0 then
          self.curNodeState = 2;
        else 
          self.curNodeState = 0;
        end
      end
      
      return true;
    end
  end
  return false;
end

function EditorTouchMagicManager:GetLastTouchMagicPath()
  return self.lastLoadedPath;
end

--判断素材是否预加载
function EditorTouchMagicManager:HasTouchMagicLoaded(path)
  if self.curNodeState == 1 and #self.touchMagicNodes > 0 and self.lastLoadedPath == path then
    return true;
  else 
    return false;
  end
end

function EditorTouchMagicManager:RemoveLastTouchMagic(magicNum, scene, removeAbsolutely, removePreNode)
  local isMagic = false;
  for i=1,magicNum do
    isMagic = self:_UnloadLastTouchMagic(scene, removeAbsolutely, removePreNode);
  end
  return isMagic;
end

function EditorTouchMagicManager:ClearTouchMagic(scene)
  for i=1,#self.touchMagicNodes do
    self:_UnloadLastTouchMagic(scene, true, true);
  end
end

function EditorTouchMagicManager:IsExistTouchMagic()
  return #self.touchMagicNodes > 0;
end

function EditorTouchMagicManager:SetMainrt(rt)
  LOG("EditorTouchMagicManager:SetMainrt")
  for index = 1, #self.touchMagicNodes do
    local touchNode = self.touchMagicNodes[index]
    self:_ConfigRenderTarget(touchNode, rt)
  end
  self.cameraRenderTarget = rt;
end

--function EditorTouchMagicManager:SetFrameIndex(frameIndex)
--  for index = #self.nodeBehaviorInstances, 1, -1 do
--    local instance = self.nodeBehaviorInstances[index]
--    if instance ~= nil then
--      instance:SetCurFrameIndex(frameIndex)
--    end
--  end
--end

function EditorTouchMagicManager:updateFrameInfo(frameIndex, pts)
  --LOG("updateFrameInfo frameIndex="..frameIndex..", pts="..pts);
  self.curVideoPlayTs = pts
  if self.needSetCacheStartTs == true then
    self:_SetCacheStartTs(self.curVideoPlayTs)
  end

  local count = #self.nodeBehaviorInstances
  for index = 1, count do
    if index < count or self.curNodeState ~= 1 then --最新结点如果是预加载的，不需要update
      local instance = self.nodeBehaviorInstances[index]
      instance:updateFrameInfo(frameIndex, pts)
    end
  end
end

function EditorTouchMagicManager:_SetCacheStartTs(timeTs)
  LOG("_SetCacheStartTs ts="..timeTs)
  self.needSetCacheStartTs = false
  local lastIndex = #self.touchCacheBehaviorInstances
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  if #instances > 0 then
    for index=1,#instances do
      instances[index]:SetStartTs(timeTs)
    end
  end
end

function EditorTouchMagicManager:StartApplyTouchMagic()  
  --收到有效的触屏信息才开始应用
  if self.lastTouchPoint == nil then
    self.needStartApply = true
    LOG("[StartApplyTouchMagic] do not start util receive available position")
    return
  end
  
  self.lastTouchPoint = nil;
  self.needStartApply = false;
  self.nodeBehaviorInstances[#self.nodeBehaviorInstances]:ResetPtsInited()
  self:_SetParticleVisibility(#self.touchCacheBehaviorInstances, true)
  self:_StartCache(true)
  self.curNodeState = 2
end

function EditorTouchMagicManager:StopApplyTouchMagic(endFrameIndex)
  self:_StartCache(false)
  self.curNodeState = 2
end

function EditorTouchMagicManager:_StartCache(start)
  LOG("EditorTouchMagicManager:_StartCache")
  local lastIndex = #self.touchCacheBehaviorInstances;
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  if #instances > 0 then
    for index=1,#instances do
      if start == true then
        self.needSetCacheStartTs = true
        instances[index]:StartCache()
      else
        instances[index]:StopCache(self.curVideoPlayTs)
      end
    end
  end
end

function EditorTouchMagicManager:SetTouchInfo(touchInfo)
  local posVec2 = touchInfo:GetTouchPoint(1)
  if posVec2:x() >= -1 and posVec2:y() >= -1 and self.touchCacheBehaviorInstances ~= nil then
    local lastIndex = #self.touchCacheBehaviorInstances;
    local instances = self.touchCacheBehaviorInstances[lastIndex]
    --LOG(string.format("[SetTouchInfo] x=%f, y=%f, lastIndex=%d",  posVec2:x(), posVec2:y(), lastIndex))
    if #instances > 0 then
      for index=1,#instances do
        instances[index]:SetTouchPosition(posVec2);

        if self.curNodeState == 1 and self.needStartApply then
          LOG("[SetTouchInfo]record position before startapply, needStartApply="..tostring(self.needStartApply))
          self.lastTouchPoint = posVec2
          self:StartApplyTouchMagic()
        end
      end
    else
      LOG("instance is nil")
    end
  end
end

function EditorTouchMagicManager:SetParticleScale(scale)
  local lastIndex = #self.touchCacheBehaviorInstances;
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  if #instances > 0 then
    for index=1,#instances do
      instances[index]:SetParticleScale(scale)
    end
  end
end

-- @paran color int类型，rgb
function EditorTouchMagicManager:SetParticleColor(color)
  local lastIndex = #self.touchCacheBehaviorInstances;
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  if #instances > 0 then
    for index=1,#instances do
      instances[index]:SetParticleColor(color)
    end
  end
end

--多个粒子大小一样吗
function EditorTouchMagicManager:GetParticleScale()
  local lastIndex = #self.touchCacheBehaviorInstances;
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  local res = 1;
  if #instances > 0 then
    res = instances[#instances]:GetParticleScale()
  end
  return res;
end

function EditorTouchMagicManager:GetParticleColor()
  local lastIndex = #self.touchCacheBehaviorInstances;
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  local res = 0;
  if #instances > 0 then
    res = instances[#instances]:GetParticleColor()
  end
  return res;
end

--预览粒子效果
function EditorTouchMagicManager:StartPreviewTouchMagic(frameIndex)
  LOG("StartPreviewTouchMagic")
  self.curNodeState = 3
  self:_SetOtherParticleVisibility(false)
  self:_PreviewCurTouchMagic(true)
end

--结束预览粒子效果
function EditorTouchMagicManager:StopPreviewTouchMagic()
  LOG("StopPreviewTouchMagic")
  self:_PreviewCurTouchMagic(false)
  self:_SetOtherParticleVisibility(true)
  self.curNodeState = 1 --停止预览，particleComponent reset了
end

--编辑页进入touch magic页面时，保存上次的touch，用于页面返回撤销
--把当前的touch magic节点的staticid存起来
function EditorTouchMagicManager:StoreTouchMagicState()
  LOG(string.format("StoreTouchMagicState touchMagicNodes size=%d", #self.touchMagicNodes));
  for index = 1, #self.touchMagicNodes do 
    table.insert(self.storeNodesId, self.touchMagicNodes[index]:GetStaticID());
  end
end

--touch magic页面撤销返回编辑页，恢复上次的touch magic
function EditorTouchMagicManager:RestoreTouchMagicState(restore, scene)
  LOG(string.format("RestoreTouchMagicState restore=%s", tostring(restore)))
  if restore then --撤销
    local remainNum = #self.storeNodesId - #self.storeTouchMagicNodes;
    local newNodesNum = #self.touchMagicNodes - remainNum;
    --1、先删除本次新加的touch magic
    self:RemoveLastTouchMagic(newNodesNum, scene, true, true); 
    --2、恢复之前的touch magic
    local restoreNum = #self.storeTouchMagicNodes;
    for index = 1, restoreNum do
      self.storeTouchMagicNodes[index].Active = true;
      table.insert(self.touchMagicNodes, remainNum+1, self.storeTouchMagicNodes[index]);
      table.insert(self.touchCacheBehaviorInstances, remainNum+1, self.storeTouchCacheBehaviorInstances[index]);
      table.insert(self.nodeBehaviorInstances, remainNum+1, self.storeNodeBehaviorInstances[index]);
    end
  else --不撤销
    -- 退出touch magic页面时，删掉预加载的结点 
    if self.curNodeState == 1 and #self.touchMagicNodes > 0 then
      self:_UnloadLastTouchMagic(scene, true, true);
    end
    
    local restoreNum = #self.storeTouchMagicNodes;
    --从场景中删掉节点
    for index = 1, restoreNum do
      local touchNode = self.storeTouchMagicNodes[index];
      self.touchMagicRootNode:DetachNode(touchNode);
      scene:DeleteNode(touchNode);
    end
  end
  self.storeNodesId = {};
  self.storeTouchMagicNodes = {};
  self.storeTouchCacheBehaviorInstances = {};
  self.storeNodeBehaviorInstances = {};
end

function EditorTouchMagicManager:_CreateTouchRootNodeIfNeed(scene)
  if self.touchMagicRootNode == nil then
    self.touchMagicRootNode = scene:CreateNode(apollocore.Node.CT_NODE);
    self.touchMagicRootNode:SetName("touchMagicRootNode");
    local component = self.touchMagicRootNode:CreateComponent(apollocore.Node.CT_TRANSFORM);
    component:SetLocalPosition(mathfunction.vector3(0.0,0.0,0.0));
  end
end

function EditorTouchMagicManager:_CreateTouchNode(cutterScene, path, rootconfig)
  if rootconfig.scene == nil then
    return nil;
  end

  -- NOTE: 创建一个新的节点，新的节点挂一个behavior,用bahavior去管理node的active状态
  local touchNode = cutterScene:CreateNode(apollocore.Node.CT_NODE);
  touchNode:SetName("Touch_"..tostring(#self.touchMagicNodes + 1));

  local component = touchNode:CreateComponent(apollocore.Node.CT_TRANSFORM);
   component:SetLocalPosition(mathfunction.vector3(0.0,0.0,0.0));

  local scriptComp = touchNode:CreateComponent(apolloengine.Node.CT_SCRIPT);
  local scriptPath = "scrs:behavior/touch_node_behavior.lua";
  local instance = self:_LoadBehavior(scriptComp, scriptPath)

  local baseobjects = self:_CreateInstanceFromBundle2(
    cutterScene,
    path,
    rootconfig.scene,
    #self.touchMagicNodes + 1,
    touchNode
  )

  if instance then
    instance:initBehaviorData(touchNode, baseobjects);
    table.insert(self.nodeBehaviorInstances, instance)
  end

  self:_CameraInit(baseobjects);
  touchNode.Active = false;

  return touchNode
end

function EditorTouchMagicManager:_CreateInstanceFromBundle2(scene, resPath, resSceneConfig, id, touchNode)
  local beginTs = venuscore.ITimerSystem:GetTimevalue()

  local pathDir = string.match(resPath, "(.+)/[^/]*%.%w+$");
  local rootdir = pathDir.."/";
  venuscore.IFileSystem:SetResourcePath(rootdir);

  local bundlepath = venuscore.IFileSystem:PathAssembly(resSceneConfig);
  local file = io.open(bundlepath, "rb");
  local str = file:read("*a");
  local skipRootNodeForScene = false;
  
  local _,objects = BundleSystem:DeserializeFromPath(bundlepath,BundleSystem.DeserializeMode.Prefab,scene);
  local cameraCom = self:_GetCameraComponent(objects);  --需要找到素材主相机
  local cacheBehaviorInstances = {};

  -- 找到root node
  for i=1,#objects do
    local nativeNode = objects[i];
    if nativeNode.GetNativeNode then
      nativeNode = objects[i]:GetNativeNode();
    end

    --LOG("------->>> node name = "..nativeNode.Name)

    touchNode:AttachNode(nativeNode);

    if nativeNode:GetComponent(apolloengine.Node.CT_PARTICLE) ~= nil then
      local scriptComp = nativeNode:GetComponent(apolloengine.Node.CT_SCRIPT)
      if not scriptComp then
        scriptComp = nativeNode:CreateComponent(apolloengine.Node.CT_SCRIPT)
      end
      local cacheBehavior = self:_LoadBehavior(scriptComp, SCRIPT_PATH)
      if cacheBehavior then
        self.touchCacheCount = self.touchCacheCount + 1
        cacheBehavior:Initialize(cameraCom, self.touchCacheCount)
        table.insert(cacheBehaviorInstances, cacheBehavior)
      end
    end
  end

  table.insert(self.touchCacheBehaviorInstances, cacheBehaviorInstances)

  local cost = venuscore.ITimerSystem:GetTimevalue() - beginTs;
  LOG("============> _CreateInstanceFromBundle2 cost " .. tostring(cost))
  return objects;
end

function EditorTouchMagicManager:_CameraInit(baseobject)
  self.resourceType = self.resourceType + 1
  local sequenceBase = self.sequenceBase
  local sequenceMax = 0
  for i=1,#baseobject do
    local cameracom = baseobject[i]:GetComponent(apolloengine.Node.CT_CAMERA);
    if cameracom then
      cameracom:SetResourceType(self.resourceType);
      local sequence = cameracom:GetSequence() + sequenceBase;
      if sequence > sequenceMax then
        sequenceMax = sequence
      end
      cameracom:SetSequence(sequence);
      cameracom:SetSequenceCulling(false);
      cameracom:SetClearFlag(0)
    end

    local renderCom = baseobject[i]:GetComponent(apolloengine.Node.CT_RENDER);
    if renderCom then
      renderCom:SetResourceType(self.resourceType);
    end

    local trailsCom = baseobject[i]:GetComponent(apolloengine.Node.CT_TRAILSRENDER);
    if trailsCom then
      trailsCom:SetResourceType(self.resourceType);
    end
  end
  self.sequenceBase = sequenceMax + 1
end

function EditorTouchMagicManager:_FindCacheBehavior(node, scriptPath)
  if not node then
    LOG("_FindCacheBehavior node is nil")
    return nil;
  end

  --判断node是否挂载transform_cache_behavior.lua
  local scriptComp = node:GetComponent(apolloengine.Node.CT_SCRIPT);
  if scriptComp then
    local cacheBehavior = scriptComp.Instances[scriptPath];
    if cacheBehavior then
      return cacheBehavior;
    else
      LOG("cacheBehavior is nil")
    end
  else
    LOG("scriptComp is nil")
  end

  return nil;
end

function EditorTouchMagicManager:_ConfigRenderTarget(node, rt)
  LOG("EditorTouchMagicManager:_ConfigRenderTarget")
  if rt ~= nil then
    local scriptComp = node:GetComponent(apolloengine.Node.CT_SCRIPT);
    if scriptComp ~= nil then
      local paraSet = scriptComp.Instances
      for scrKey,scrValue in pairs(paraSet) do
        if scrValue.ConfigRenderTarget then
          scrValue:ConfigRenderTarget(rt);
        else
          LOG("ConfigRenderTarget is nil")
        end
      end
    else
      LOG("scriptComp is nil")
    end
  end
end

function EditorTouchMagicManager:_LoadBehavior(scriptComp, scriptPath)
  --LOG("[_LoadBehavior] to load  "..scriptPath)
  local instance = venuscore.LoadBehavior(scriptPath);
  if instance then
    scriptComp:InsertInstance(instance, scriptPath);
  else
    LOG("[_LoadBehavior] load failed. "..scriptPath)
  end
  return instance;
end

function EditorTouchMagicManager:_GetCameraComponent(objects)
  for i=1,#objects do
    local nativeNode = objects[i];
    if nativeNode.GetNativeNode then
      nativeNode = objects[i]:GetNativeNode();
    end

    local com = nativeNode:GetComponent(apolloengine.Node.CT_CAMERA)
    if com ~= nil then
      return com
    end
  end

  return nil
end

--判断是否是上次添加的touch magic
function EditorTouchMagicManager:_IsStoreNode(node) 
  local staticId = node:GetStaticID();
  for index = 1, #self.storeNodesId do
    if self.storeNodesId[index] == staticId then
      return true;
    end
  end
  return false;
end

--预览当前粒子效果
function EditorTouchMagicManager:_PreviewCurTouchMagic(start)
  local lastIndex = #self.touchCacheBehaviorInstances
  local instances = self.touchCacheBehaviorInstances[lastIndex]
  if #instances > 0 then
    for index=1, #instances do
      if start then
        instances[index]:StartPreviewTouchMagic()
      else
        instances[index]:StopPreviewTouchMagic()
      end
    end
  end
end

--设置指定touch节点设置显示/隐藏
function EditorTouchMagicManager:_SetParticleVisibility(id, show)
  local instances = self.touchCacheBehaviorInstances[id]
  if #instances > 0 then
    for index=1, #instances do
      instances[index]:SetParticleVisibility(show)
    end
  end
end

--除了当前touch节点，其他touch设置显示/隐藏
function EditorTouchMagicManager:_SetOtherParticleVisibility(show)
  for id = 1, #self.touchCacheBehaviorInstances -1 do
    self:_SetParticleVisibility(id, show)
  end
end

return EditorTouchMagicManager;