local venuscore = require "venuscore"
local mathfunction = require "mathfunction"
local apolloengine = require "apolloengine"
local venusjson = require "venusjson"
local libmasquerade = require "libmasquerade"
local FrameCountdown = require "behavior.tracking_behavior.frame_countdown"

local MeshSwitch = venuscore.VenusBehavior:extend("MeshSwitchBehavior");

function MeshSwitch:new()
  self.config = ""
  self.delay_countdown = FrameCountdown(30)
  self.freeze_countdown = FrameCountdown(30)
  self.unfreeze_countdown = FrameCountdown(10)
  self.inited = false
end

function MeshSwitch:GetTracking()
  return self.tracking
end

function MeshSwitch:SetTracking(tracking)
  self.tracking = tracking
end

function MeshSwitch:GetConfig()
  return self.config
end

function MeshSwitch:SetConfig(config)
  self.inited = false
  self.config = config
end

function MeshSwitch:GetDelayFrames()
  return self.delay_countdown:GetCount()
end

function MeshSwitch:SetDelayFrames(delay_frames)
  self.delay_countdown:SetCount(delay_frames)
end

function MeshSwitch:_Play()

end

function MeshSwitch:_OnAwake()

end

function MeshSwitch:_OnStart()
end

function MeshSwitch:_OnUpdate(timespan)
  if self.tracking == nil or self.config == "" then
    return
  end
  
  if not self.inited then
    self:_Init()
  end
  
  if not self.tracking:IsTracked() then
    return
  end
  
  if self.animation == nil then
    LOG("animation null")
    return
  end
  
  local next_target = self:_GetSatisfiedTarget()    
  local playing = self.animation:GetAnimationStatus()
  if playing ~= apolloengine.IComponent.AS_PLAYING then
    if self.show_target ~= next_target then
      if next_target then
        self:_ActivateTargetNodes(next_target, true)
        self.freeze_countdown:Start()
      end
      if self.show_target then
        self:_ActivateTargetNodes(self.show_target, false)
        self.unfreeze_countdown:Start()
      end
      self.show_target = next_target
    end      
          
    if next_target then
      self:_ActivateBase(true, next_target.freeze, next_target.mouthbs)
      self.animation:Reset()
      self.animation:Play()
    else
      self:_ActivateBase(false, true, -1)
    end
  else
    if next_target then
      self:_UpdateMouthBS(next_target.mouthbs)
    end
  end
end

function MeshSwitch:_Break(value, interval)
  if value < interval[1] or value > interval[2] then
    return true
  end
  return false
end

function MeshSwitch:_Init()
  local config_str = venusjson.LaodJsonFile(self.config)
  self.base = self.Node:GetNodeByName(config_str.base)
  
  if config_str.attach ~= nil then
    self.attach = {node = self.Node:GetNodeByName(config_str.attach.name)}
    self.attach.script_behavior = nil
    local script_component = self.attach.node:GetComponent(apolloengine.Node.CT_SCRIPT);
    if script_component ~= nil then
      for script_key, script_value in pairs(script_component.Instances) do
        if script_key == config_str.attach.script then
          self.attach.script_behavior = script_value
          break
        end
      end
    end
  end

  if config_str.animation ~= nil then
    self.animation = self.Node:GetNodeByName(config_str.animation.name):GetComponent(apolloengine.Node.CT_ANIMATION)
    for _, name in ipairs(config_str.animation.ignore) do
      self.animation:SetIgnoreAnim(name)
    end
  end
  
  self.targets = {}
  for _, v in ipairs(config_str.targets) do
    local node_list = {}
    for _, name in ipairs(v.name) do
      local node = self.Node:GetNodeByName(name)
      node.Active = false
      table.insert(node_list, node)
    end
    local target = {nodes = node_list, condition = v.condition, freeze = v.freeze, mouthbs = v.mouthbs}
    table.insert(self.targets, target)
  end
  
  self.inited = true
end

function MeshSwitch:_GetBsWeight()
  local num = self.tracking:GetBlendshapeNumber()
  local weight_dict = {}
  for i = 1, num do 
    local weight = self.tracking:GetBlendshapeWeight(i - 1)
    local name = self.tracking:GetBlendshapeName(i - 1)
    weight_dict[name] = weight
  end
  return weight_dict
end

function MeshSwitch:_GetSatisfiedTarget()
  local weight_dict = self:_GetBsWeight()
  local next_target = nil
  for _, target in ipairs(self.targets) do
    local show = true
    for name, interval in pairs(target.condition) do
      local weight = weight_dict[name] or 0
      if self:_Break(weight, interval) then
        show = false
        break
      end
    end
    if show then
      next_target = target
      break
    end
  end
  
  if next_target ~= nil then
    if next_target == self.candidata_target then
      if self.delay_countdown:Finished() then
        return next_target
      end
      self.delay_countdown:Update()
    else
      self.delay_countdown:Start()
    end
  end
  
  self.candidata_target = next_target
  return nil
end

function MeshSwitch:_ActivateBase(active, freeze, mouthbs_id)
  self.base.Active = not active
  if freeze == true and self.attach.script_behavior ~= nil then
    self.attach.script_behavior:Freeze(active)
    if mouthbs_id >= 0 then
      self:_UpdateMouthBS(mouthbs_id)
    else
      self:_ResetMouthBS()
    end
  end
end

function MeshSwitch:_ActivateTargetNodes(target, active)
  for _, node in ipairs(target.nodes) do
    node.Active = active
  end
end

function MeshSwitch:_UpdateMouthBS(mouthbs_id)
  local morph_component = self.attach.node:GetComponent(apolloengine.Node.CT_MORPH)
  if not self.freeze_countdown:Finished() then
    local progress = self.freeze_countdown:Progress()
    morph_component:SetChannelPercents(0, mouthbs_id, 2.5 * progress)

    -- local step = 1 / self.freeze_countdown:GetCount()
    -- for i = 1, 20 do
    --   local bs_value = morph_component:GetChannelPercents(1, i-1)
    --   local ori_value = bs_value / (1 - (progress - step))
    --   morph_component:SetChannelPercents(1, i-1, ori_value * (1 - progress))
    -- end
    self.freeze_countdown:Update()
  else
    morph_component:SetChannelPercents(0, mouthbs_id, 100.0)
  end
end

function MeshSwitch:_ResetMouthBS()
  local morph_component = self.attach.node:GetComponent(apolloengine.Node.CT_MORPH)
  if not self.unfreeze_countdown:Finished() then
    local progress = self.unfreeze_countdown:Progress()
    for i = 1, 8 do
      local val = morph_component:GetChannelPercents(0, i-1)
      if val > 0.0 then
        morph_component:SetChannelPercents(0, i-1, 100.0 * (1 - progress))
      end
    end
    self.unfreeze_countdown:Update()
  else
    for i = 1, 8 do
      morph_component:SetChannelPercents(0, i-1, 0.0)
    end
  end
end


MeshSwitch:MemberRegister("tracking",
    venuscore.ScriptTypes.ReferenceType(
        libmasquerade.TrackingComponent:RTTI(),    
        MeshSwitch.GetTracking,
        MeshSwitch.SetTracking
))

MeshSwitch:MemberRegister("config",  
  venuscore.ScriptTypes.FilePathType(
    {"json"},
    MeshSwitch.GetConfig,
    MeshSwitch.SetConfig
  ))

MeshSwitch:MemberRegister("delay frames",
  venuscore.ScriptTypes.IntType(
    0, 300,
    MeshSwitch.GetDelayFrames,
    MeshSwitch.SetDelayFrames
));

return MeshSwitch