require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local NoseShin = {}

function NoseShin:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function NoseShin:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function NoseShin:_InitParams(defined)
    beautydefined = defined
    -- 瘦鼻参数
    self.coef = self.coef or 0
    self.coef_temp = 1.0;
    -- shader params
    apolloengine.ShaderEntity.UNIFORM_NOSECOEF =
        apolloengine.IMaterialSystem:NewParameterSlot(
            apolloengine.ShaderEntity.UNIFORM, "UNIFORM_NOSECOEF");
    apolloengine.ShaderEntity.NOSEPOINTS =
        apolloengine.IMaterialSystem:NewParameterSlot(
            apolloengine.ShaderEntity.UNIFORM, "NOSEPOINTS");
    apolloengine.ShaderEntity.NOSEPARAMS =
        apolloengine.IMaterialSystem:NewParameterSlot(
            apolloengine.ShaderEntity.UNIFORM, "NOSEPARAMS");
end

function NoseShin:SetParams(config, shaderParams)
    shaderParams.nosepointArray = mathfunction.vector4array();
    shaderParams.noseParaArray = mathfunction.vector2array();

    shaderParams.noseParaArray:PushBack(beautyutils._ArrayToVec2(
                                            config.noseparam[1]));
    shaderParams.noseParaArray:PushBack(beautyutils._ArrayToVec2(
                                            config.noseparam[1]));
    shaderParams.noseParaArray:PushBack(beautyutils._ArrayToVec2(
                                            config.noseparam[2]));
    shaderParams.noseParaArray:PushBack(beautyutils._ArrayToVec2(
                                            config.noseparam[2]));
                                        
    self:_InitLens(config)
end

function NoseShin:_InitLens(config)
  self.lens = {}
  for i = 1, #beautydefined.NoseIndices do
    local index1 = beautydefined.NoseIndices[i][1] + 1
    local index2 = beautydefined.NoseIndices[i][2] + 1;
    self.lens[#self.lens + 1] = Len(index1, index2, config.noseparam[i][1], 0, config.noseparam[i][2])
    self.lens[#self.lens + 1] = Len(index2, index1, config.noseparam[i][1], 0, config.noseparam[i][2])
  end
end

function NoseShin:_InitQuad(quadnode, shaderParams)
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSECOEF,
                          mathfunction.vector1(0.0));
    quadnode:SetParameter(apolloengine.ShaderEntity.NOSEPARAMS,
                          shaderParams.noseParaArray);
    quadnode:SetParameter(apolloengine.ShaderEntity.NOSEPOINTS,
                          shaderParams.nosepointArray);
end

function NoseShin:_DrawQuad(quadnode, keypoints, rate)
    local nosepointArray = mathfunction.vector4array();
    for i = 1, 2, 1 do
        local index1 = beautydefined.NoseIndices[i][1] + 1;
        local index2 = beautydefined.NoseIndices[i][2] + 1;
        nosepointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1],
                                                         keypoints[index2]));
        nosepointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index2],
                                                         keypoints[index1]));
    end
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSECOEF,
                          mathfunction.vector1(self.coef * rate));
    quadnode:SetParameter(apolloengine.ShaderEntity.NOSEPOINTS, nosepointArray);
end

function NoseShin:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef * rate
  local tmpDis = {}
  
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)  
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  for i = 1, #beautydefined.NoseWarpMap do
    local src = beautydefined.NoseWarpMap[i][1] + 1
    local dst = beautydefined.NoseWarpMap[i][2] + 1
    keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.NoseWarpCoef[i]
    keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.NoseWarpCoef[i]
  end
end

function NoseShin:UpdateParaValue(quadnode, config, key_index,
                                  attribute, value)
    if (attribute == "X") then
        config.noseparam[key_index - beautydefined.nose_start + 1][1] = value;
        LOG("nose param " .. key_index - 15 .. " x set to: " .. value);
    end
    if (attribute == "R") then
        config.noseparam[key_index - beautydefined.nose_start + 1][2] = value;
        LOG("nose param " .. key_index - 16 .. " R set to: " .. value);
    end
    local shaderParams = {}
    self:SetParams(config, shaderParams)
    quadnode:SetParameter(apolloengine.ShaderEntity.NOSEPARAMS,
                          shaderParams.noseParaArray);
end

function NoseShin:UpdateCoeff(quadnode, value, rate)
    if (value == 1) then
        self.coef = 1.0;
    else
        self.coef = 0.0;
    end
    LOG("Nose Shin coef set to: " .. self.coef);
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSECOEF,
                          mathfunction.vector1(self.coef * rate));
end

function NoseShin:GetParams(config) return beautyutils.Clone(config.noseparam); end

function NoseShin:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef
  --WARNING("[facelift]: Nose Shin coef: " .. self.coef)
end

function NoseShin:Release()
  self.lens = {}
end

return NoseShin
