require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.ellipselen"

local EyeScale = {}

function EyeScale:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function EyeScale:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
    self.coef = self.coef_temp;
	end  
end

function EyeScale:_InitParams(defined)
  beautydefined = defined
  --大眼参数
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.EYESCALE_PARAM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"EYESCALE_PARAM");
  apolloengine.ShaderEntity.UNIFORM_EYECOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYECOEF");
  apolloengine.ShaderEntity.EYECORNERPOINTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"EYECORNERPOINTS");
end

function EyeScale:SetParams(config, shaderParams)
  shaderParams.eyescaleParaArray = mathfunction.vector2array();
  shaderParams.eyescaleParaArray:PushBack(beautyutils._ArrayToVec2(config.EyescaleRadius));
  shaderParams.eyescaleParaArray:PushBack(beautyutils._ArrayToVec2(config.EyescaleParam));
  
  shaderParams.EyeCornerArray = mathfunction.vector2array();
  self:_InitLens(config)
end

function EyeScale:_InitLens(config)
  self.lens = {}
  if config.likee_version then
  else
    self.lens[#self.lens + 1] = Len(beautydefined.LeftEyeCornerIndices, config.EyescaleParam, config.EyescaleRadius)
    self.lens[#self.lens + 1] = Len(beautydefined.RightEyeCornerIndices, config.EyescaleParam, config.EyescaleRadius)
  end
end

function EyeScale:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.EYESCALE_PARAM, shaderParams.eyescaleParaArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYECOEF, mathfunction.vector1(0.0));
  quadnode:SetParameter(apolloengine.ShaderEntity.EYECORNERPOINTS, shaderParams.EyeCornerArray);
end

function EyeScale:_DrawQuad(quadnode, keypoints, rate, config) 		
  local EyeCornerArray = mathfunction.vector2array(); 
  for i = 1, #beautydefined.LeftEyeCornerIndices, 1 do
    EyeCornerArray:PushBack(beautyutils._ArrayToVec2(keypoints[beautydefined.LeftEyeCornerIndices[i] + 1]));
  end
  for i = 1, #beautydefined.RightEyeCornerIndices, 1 do
    EyeCornerArray:PushBack(beautyutils._ArrayToVec2(keypoints[beautydefined.RightEyeCornerIndices[i] + 1]));
  end
  
  quadnode:SetParameter(apolloengine.ShaderEntity.EYECORNERPOINTS, EyeCornerArray); 
  --LOG("for facebeauty project");
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYECOEF,mathfunction.vector1(self.coef * rate));
end

function EyeScale:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef * rate
  for i = 1, #self.lens do
    local cornerIdx = self.lens[i]:GetCornerIdx()
    local kp1 = keypoints[cornerIdx[1] + 1];
    local kp2 = keypoints[cornerIdx[2] + 1];
    local kp3 = keypoints[cornerIdx[3] + 1];
    local kp4 = keypoints[cornerIdx[4] + 1];

    self.lens[i]:InitEyeParams(kp1, kp2, kp3, kp4, facedir, coef, aspect_ratio) 
  end

  local indexRange = self:_getIndexRange(#keypoints)
  for i = 1, #indexRange do
    local index = indexRange[i] + 1
    for j = 1, #self.lens do
      local tmp = self.lens[j]:WarpObliqueEllipseScale(keypoints[index], aspect_ratio)
      keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmp[1]
      keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmp[2]
    end
  end
end

function EyeScale:_getIndexRange(num)
  local min = 30
  local max = 79
  local range= {0, 1, 2, 104, 105}
  for i = min, max do
    table.insert(range, i)
  end
  
  if num >= 240 then
    min = 0
    max = 43
    for i = min, max do
      table.insert(range, i + 106)
    end
  end
  return range
end

function EyeScale:UpdateParaValue(quadnode, config, key_index, attribute, value)
  
  if(attribute == "Inten") then
    config.EyescaleParam[1] = value;
  end
  if (attribute == "YScale") then
    config.EyescaleParam[2] = value;
  end
  if(attribute == "RR") then
    config.EyescaleRadius[1] = value;
  end
  if (attribute == "rr") then
    config.EyescaleRadius[2] = value;
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.EYESCALE_PARAM, shaderParams.eyescaleParaArray);
end

function EyeScale:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0; -- full warping
  else
    self.coef = 0.0; -- no warping
  end
  LOG("Eye Scale coef set to: " .. self.coef);
  
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYECOEF, mathfunction.vector1(self.coef*rate));
end

function EyeScale:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --WARNING("[facelift]: Eye Scale coef: " .. self.coef)
end

function EyeScale:GetParams(config)
  return {config.EyescaleRadius[1], config.EyescaleRadius[2], config.EyescaleParam[1], config.EyescaleParam[2]};
end

function EyeScale:Release()
  self.lens = {}
end
  
return EyeScale