require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local EyeBrow = {}

function EyeBrow:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function EyeBrow:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function EyeBrow:_InitParams(defined)
  beautydefined = defined
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_EYEBROWCOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEBROWCOEF");
  apolloengine.ShaderEntity.EYEBROWPARA = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"EYEBROWPARA");
  apolloengine.ShaderEntity.EYEBROWPARARADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"EYEBROWPARARADIUS");
  apolloengine.ShaderEntity.EYEBROWPOINTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"EYEBROWPOINTS");
end

function EyeBrow:SetParams(config, shaderParams)
  shaderParams.eyebrowpointArray = mathfunction.vector4array();
  shaderParams.eyebrowParaArray = mathfunction.vector2array();
  shaderParams.eyebrowParamRadiusArray = mathfunction.vector2array();

  for i = 1, #beautydefined.EyebrowKey, 1 do
    if type(beautydefined.EyebrowKey[i]) == "table" then
      shaderParams.eyebrowParaArray:PushBack(beautyutils._ArrayToVec2(config.EyebrowParam[i]));
      shaderParams.eyebrowParaArray:PushBack(beautyutils._ArrayToVec2(config.EyebrowParam[i]));
    end
  end
  
  for i = 1, #beautydefined.EyebrowKey, 1 do
    if type(beautydefined.EyebrowKey[i]) == "table" then
      shaderParams.eyebrowParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.EyebrowParamRadius[i]));
      shaderParams.eyebrowParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.EyebrowParamRadius[i]));
    end
  end
  self:_InitLens(config)
end

function EyeBrow:_InitLens(config)
  self.lens = {}
  for i = 1, #beautydefined.EyebrowKey do
    local index1 = beautydefined.EyebrowKey[i][1]+1;
    local index2 = beautydefined.EyebrowDir[i][2]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, beautydefined.EyebrowParamScript[i][1], beautydefined.EyebrowParamScript[i][2], beautydefined.EyebrowParamRadiusScript[i])
    
    index1 = beautydefined.EyebrowKey[i][2]+1;
    index2 = beautydefined.EyebrowDir[i][1]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, beautydefined.EyebrowParamScript[i][1], beautydefined.EyebrowParamScript[i][2], beautydefined.EyebrowParamRadiusScript[i])
  end
end

function EyeBrow:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPOINTS, shaderParams.eyebrowpointArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPARA, shaderParams.eyebrowParaArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPARARADIUS, shaderParams.eyebrowParamRadiusArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEBROWCOEF, mathfunction.vector1(0.0));
end

function EyeBrow:_DrawQuad(quadnode, keypoints, rate)
  local eyebrowpointArray = mathfunction.vector4array();
  for i = 1, #beautydefined.EyebrowKey do
    if type(beautydefined.EyebrowKey[i])=="table" then
        local index1 = beautydefined.EyebrowKey[i][1]+1;
        local index2 = beautydefined.EyebrowDir[i][2]+1;
        eyebrowpointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
        
        index1 = beautydefined.EyebrowKey[i][2]+1;
        index2 = beautydefined.EyebrowDir[i][1]+1;
        eyebrowpointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
    end
  end
  
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEBROWCOEF,mathfunction.vector1(-self.coef * rate));
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPOINTS,eyebrowpointArray);
end

function EyeBrow:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = -self.coef * rate
  local tmpDis = {}
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  for i = 1, #beautydefined.EyebrowWarpMap do
    local src = beautydefined.EyebrowWarpMap[i][1] + 1
    local dst = beautydefined.EyebrowWarpMap[i][2] + 1
    keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.EyebrowWarpCoef[i]
    keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.EyebrowWarpCoef[i]
  end
  
  if #keypoints >= 240 then
    for i = 1, #beautydefined.AdvEyebrowWarpMap do
      local src = beautydefined.AdvEyebrowWarpMap[i][1] + 107
      local dst = beautydefined.AdvEyebrowWarpMap[i][2] + 1
      keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.AdvEyebrowWarpCoef[i]
      keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.AdvEyebrowWarpCoef[i]
    end
  end
end

function EyeBrow:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if(attribute == "X") then
      config.EyebrowParam[key_index - beautydefined.Eyebrow_start + 1][1] = value;
    --      LOG("FaceParam[" .. self.key_param_table[key_index] .. "][1] set to " .. value);
    end
    if(attribute == "Y") then
      config.EyebrowParam[key_index - beautydefined.Eyebrow_start + 1][2] = value;
    --      LOG("FaceParam[" .. self.key_param_table[key_index] .. "][2] set to " .. value);
    end 
    if(attribute == "R") then
      config.EyebrowParamRadius[key_index - beautydefined.Eyebrow_start + 1][1] = value;
    --      LOG("FaceParamRadius[" .. self.key_param_table[key_index] .. "][1] set to " .. value);
      if(config.EyebrowParamRadius[key_index - beautydefined.Eyebrow_start + 1][1] < 0.1) then
        config.EyebrowParamRadius[key_index - beautydefined.Eyebrow_start + 1][1] = 0.1;
      end
    end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPARA, shaderParams.eyebrowParaArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.EYEBROWPARARADIUS, shaderParams.eyebrowParamRadiusArray);
end

function EyeBrow:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0.0;
  end
  LOG("Eyebrow coef set to: " .. self.coef);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEBROWCOEF,mathfunction.vector1(-self.coef * rate));
end

function EyeBrow:GetParams(config)
  local EyebrowParam = beautyutils.Clone(config.EyebrowParam);
  local EyebrowParamRadius = beautyutils.Clone(config.EyebrowParamRadius);
  return EyebrowParam, EyebrowParamRadius
end

function EyeBrow:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Eyebrow: " .. self.coef)
end

function EyeBrow:Release()
  self.lens = {}
end
  
return EyeBrow