local an = require "apolloutility.apollonode"
local ae = require "apolloengine"
local mf = require "mathfunction"
local vc = require "venuscore"

local defines =
{
  blitterCount = 100,
  seqInterval = 100,
  blitterMaterial = "comm:documents/material/imageblit.material",
}
-- LayerID:     Sticker
-- SubLayerID:  Sticker-2
local LAYERS =
{
  "Input",
  "EditorMakeUp",
  "MakeUp",
  "FaceBeautify",
  "FaceLift",
  "Sharpen",
  "Stylish",
  "Sticker",
  "LiveGame",
  "Output",
}

local function parseLayerID(id)
  if id == nil then
    return nil, nil
  end
  local k = string.find(id, "-")
  if k == nil then
    return id, nil
  end
  local baseID = string.sub(id, 1, k - 1)
  local layerOrder = tonumber(string.sub(id, k + 1))
  return baseID, layerOrder
end

local function getNodeID(node)
  if node.node then
    return node.node:GetObjectID()
  else
    return node:GetObjectID()
  end
end

local function setNodeSeq(node, seq)
  if node.SetSequence then
    node:SetSequence(seq)
  else
    local render = node:GetComponent(ae.Node.CT_RENDER)
    if render then
      render:SetSequence(seq)
    end
  end
end

local RenderQueue = {}

function RenderQueue:Initialize(mainCamera)
  self:_InitRootNodes()
  self:_InitLayers(LAYERS)
  self:_InitBlitters(defines.blitterCount, mainCamera)
  self:_InitCameraPool(defines.blitterCount, mainCamera)
  self.subLayerObjs = {}
  -- The only purpose of keeping this thing
  -- is to ensure deformation below to facechange
  self.facemaxz = -2000
end

function RenderQueue:_InitRootNodes()
  self.rootnode = an.VirtualNode();
  self.rootnode:SetName("Queue Camera Root");
  self.cameraroot = an.VirtualNode();
  self.cameraroot:SetName("Camera Root");
  self.renderroot = an.VirtualNode();
  self.renderroot:SetName("Render Root");
  self.rootnode:AttachNode(self.cameraroot);
  self.rootnode:AttachNode(self.renderroot);
end

function RenderQueue:_InitLayers(layerIDs)
  self.layers = {}
  layerIDs = { unpack(layerIDs) }
  for i = 1, #layerIDs do
    self.layers[i] =
    {
      id = layerIDs[i],
      index = i,
      nodes = setmetatable({}, {__mode = "v"}),
      active = false,
      blitter = nil,
      blitterCount = 0,
      cameras = {},
      insertedCameraQueues = {},
      insertedCameraCount = 0,
    }
  end
end

function RenderQueue:_InitBlitters(count, finalCamera)
  self.blitters = {}
  local swaps =
  {
    ae.RenderTargetEntity.ST_SWAP_E,--标记作用，主要用于底层资源共享
    ae.RenderTargetEntity.ST_SWAP_F
  }
  local seq = finalCamera:GetSequence() - (count - 1) * defines.seqInterval
  for i = 1, count do
    local rt = nil
    local tex = nil
    if i < count then
      rt, tex = self:_CreateBlitterRT(swaps[(i - 1) % 2 + 1])
    end
    local render = self:_CreateBlitterRender(seq)
    self.renderroot:AttachNode(render);
    self.blitters[i] =
    {
      active = false,
      sequence = seq,
      renderTarget = rt,
      texture = tex,
      render = render,
      prev = nil,
      next = nil,
    }
    seq = seq + defines.seqInterval
  end
  for i = 1, count do
    local blitter = self.blitters[i]
    if i > 1 then
      blitter.prev = self.blitters[i - 1]
    end
    if i < count then
      blitter.next = self.blitters[i + 1]
    end
  end
end

function RenderQueue:_InitCameraPool(count, mainCamera)
  self.cameras = {}
  local near = mainCamera:GetNear();
  local far = mainCamera:GetFar();
  local pos = mainCamera:GetWorldPosition();
  local look = pos + mainCamera:GetForward();
  local up = mainCamera:GetUp();
  for i = 1, count - 1 do
    local camera = self:_CreateCamera(
      i,
      near,
      far,
      pos,
      look,
      up
    )
    self.cameraroot:AttachNode(camera)
    table.insert(self.cameras, camera)
  end
  table.insert(self.cameras, mainCamera)
end

function RenderQueue:_CreateBlitterRT(swap)
  local rt = ae.RenderTargetEntity();--创建一个FBO
  rt:PushMetadata(--设置FBO格式
    ae.RenderTargetMetadata(
      ae.RenderTargetEntity.RT_RENDER_TARGET_2D,
      swap,--标记作用，主要用于底层资源共享
      ae.Framework:GetViewport(),
      ae.Framework:GetResolution()
    )
  );--分辨率
  local depth = rt:MakeTextureAttachment(ae.RenderTargetEntity.TA_DEPTH_STENCIL);--增加深度纹理
  depth:PushMetadata(
    ae.DepthRenderBufferMetadata(
      swap,
      ae.Framework:GetResolution(),
      ae.TextureEntity.PF_DEPTH24_STENCIL8
    ));
  local tex = rt:MakeTextureAttachment(ae.RenderTargetEntity.TA_COLOR_0);--增加color0纹理
  tex:PushMetadata(--创建纹理
    ae.TextureRenderMetadata(
      swap,--此处的纹理swap和尺寸必须和rt的相同不然将导致未定义的错误
      ae.Framework:GetResolution()
    )
  );
  rt:CreateResource();
  return rt, tex;
end

function RenderQueue:_CreateCamera(index, near, far, pos, look, up)
  local camera = an.CameraNode();--新建摄像机
  camera:SetName("Queue Camera "..tostring(index));
  camera:CreateRealCameraProjection(near, far);--设置摄像机
  camera:LookAt(pos, look, up);
  camera:Recalculate();
  return camera
end

function RenderQueue:_CreateBlitterRender(seq)
  local render = an.QuadNode()
  render:CreateResource(defines.blitterMaterial, true)
  render:SetSequence(seq)
  render:SetCull(false)
  render:SetShow(false)
  render:SetName("Queue Quad "..tostring(seq));
  return render
end

-- function RenderQueue:_CreateDefaultSourceTexture()
--   local tex = ae.TextureEntity()
--   tex:PushMetadata(
--     ae.TextureBufferMetadata(
--       mf.vector2(2, 2)
--     )
--   )
--   tex:CreateResource()
--   return tex
-- end

function RenderQueue:_ObtainCamera()
  local camera = self.cameras[#self.cameras]
  table.remove(self.cameras)
  camera:Activate()
  return camera
end

function RenderQueue:_ReleaseCamera(camera)
  camera:Deactivate()
  local rt = camera:GetAttachedRenderTarget()
  if rt ~= nil then
    camera:DetachRenderTarget(rt)
  end
  table.insert(self.cameras, camera)
end

function RenderQueue:_FindLayer(layerID)
  local baseID, order = parseLayerID(layerID)
  if baseID == nil then
    return nil, nil
  end
  for i = 1, #self.layers do
    local layer = self.layers[i]
    if layer.id == baseID then
      if order ~= nil and order > layer.blitterCount then
        return nil, nil
      else
        return layer, order
      end
    end
  end
  return nil, nil
end

--Get the last texture of this layer
function RenderQueue:GetTexture(layerID)
  local layer = self:_FindLayer(layerID)
  if layer == nil then
    return nil
  end
  if not layer.active then
    return nil
  end
  local blitter = layer.blitter
  local count = layer.blitterCount
  for k = 1, count - 1 do
    blitter = blitter.next
  end
  return blitter.texture
end

function RenderQueue:GetCamera(layerID)
  local layer, order = self:_FindLayer(layerID)
  if layer == nil then
    return nil
  end
  if not layer.active then
    return nil
  end
  return layer.cameras[#layer.cameras]
end

function RenderQueue:GetLastCamera()
  local layer = nil
  for i = #self.layers, 1, -1 do
    if self.layers[i].active then
      layer = self.layers[i]
      break
    end
  end
  if layer == nil then
    return nil
  end
  return layer.cameras[#layer.cameras]
end

function RenderQueue:_GetPrevActiveLayer(index)
  for i = index - 1, 1, -1 do
    if self.layers[i].active then
      return self.layers[i]
    end
  end
  return nil
end

function RenderQueue:_GetNextActiveLayer(index)
  for i = index + 1, #self.layers do
    if self.layers[i].active then
      return self.layers[i]
    end
  end
  return nil
end

--return sub layer id instead of base id
function RenderQueue:GetLink(layerID)
  local layer, order = self:_FindLayer(layerID)
  if layer == nil then
    return nil
  end
  if not layer.active then
    return nil
  end
  if order == nil then
    order = 1
  end
  if order > 1 then
    return string.format("%s-%d", layer.id, order - 1)
  else
    local prev = self:_GetPrevActiveLayer(layer.index)
    return prev ~= nil
      and string.format("%s-%d", prev.id, prev.blitterCount)
      or nil
  end
end

function RenderQueue:GetLinkedTexture(layerID)
  local layer, order = self:_FindLayer(layerID)
  if layer == nil then
    return nil
  end
  if not layer.active then
    return nil
  end
  if order == nil or order == 1 then
    local prevLayer = self:_GetPrevActiveLayer(layer.index)
    if prevLayer == nil then
      return nil
    end
    if prevLayer.insertedCameraCount > 0 then
      local cameraQueues = prevLayer.insertedCameraQueues
      return cameraQueues[#cameraQueues].fnGetTex()
    end
    local blitter = prevLayer.blitter
    for i = 1, prevLayer.blitterCount - 1 do
      blitter = blitter.next
    end
    return blitter.texture
  else
    local prevOrder = order - 1
    local blitter = layer.blitter
    for i = 1, prevOrder - 1 do
      blitter = blitter.next
    end
    return blitter.texture
  end
end

function RenderQueue:IsActive(layerID)
  local layer = self:_FindLayer(layerID)
  if layer == nil then
    return nil
  end
  return layer.active
end

function RenderQueue:Activate(layerID)
  local layer = self:_FindLayer(layerID)
  if layer == nil then
    return false
  end
  if not layer.active then
    layer.active = true
    layer.blitterCount = 1
    self:_Update()
  end
  return true
end

function RenderQueue:Deactivate(layerID)
  local layer = self:_FindLayer(layerID)
  if layer == nil then
    return false
  end
  if layer.active then
    layer.active = false
    layer.blitterCount = 0
    self:_Update()
  end
  return true
end

function RenderQueue:Queue(layerID, node)
  local layer = self:_FindLayer(layerID)
  if layer == nil then
    return false
  end
  layer.nodes[getNodeID(node)] = node
  if layer.active then
    setNodeSeq(node, layer.blitter.sequence)
  else
    layer.active = true
    layer.blitterCount = 1
    self:_Update()
  end
  return true
end

function RenderQueue:Prev(node)
  return self:Queue("Input", node)
end

function RenderQueue:After(node)
  return self:Queue("Output", node)
end

-- function RenderQueue:Clear(layerIDStart, layerIDEnd)
--   local layer0 = self:_FindLayer(layerIDStart)
--   local layer1 = self:_FindLayer(layerIDEnd)
--   for i = layer0.index, layer1.index - 1 do
--     local layer = self.layers[i]
--     layer.active = false
--   end
--   self:_Update()
-- end

function RenderQueue:_CheckLayerNodes(layer)
  for id, node in pairs(layer.nodes) do
    if vc.isNil(node.node) then
      layer.nodes[id] = nil
    end
  end
end

function RenderQueue:_ReLinkLayer(layer, nextLayer)
  -- Firstly, link all blitters belonged to the layer
  local blitter = layer.blitter
  for k = 1, layer.blitterCount - 1 do
    blitter.next.render:SetParameter(
      ae.ShaderEntity.TEXTURE_DIFFUSE,
      blitter.texture
    )
    blitter = blitter.next
  end
  if nextLayer == nil then
    return
  end
  -- Then link inserted cameras
  local prevCameraQueue = nil
  for i = 1, #layer.insertedCameraQueues do
    local cameraQueue = layer.insertedCameraQueues[i]
    if i == 1 then
      cameraQueue.fnSetTex(blitter.texture)
    else
      cameraQueue.fnSetTex(prevCameraQueue.fnGetTex())
    end
    prevCameraQueue = cameraQueue
  end
  -- At last link to nextLayer
  if prevCameraQueue ~= nil then
    nextLayer.blitter.render:SetParameter(
      ae.ShaderEntity.TEXTURE_DIFFUSE,
      prevCameraQueue.fnGetTex()
    )
  else
    nextLayer.blitter.render:SetParameter(
      ae.ShaderEntity.TEXTURE_DIFFUSE,
      blitter.texture
    )
  end
end

function RenderQueue:_ReSeqLayer(layer)
  local blitters = {}
  local blitter = layer.blitter
  for k = 1, layer.blitterCount do
    local camera = layer.cameras[k]
    local rt = camera:GetAttachedRenderTarget()
    if rt ~= nil then
      camera:DetachRenderTarget(rt)
    end
    if blitter.renderTarget ~= nil then
      camera:AttachRenderTarget(blitter.renderTarget)
    end
    camera:SetSequence(blitter.sequence)
    table.insert(blitters, blitter)
    blitter = blitter.next
  end
  for _, node in pairs(layer.nodes) do
    local layerOrder = node.layerOrder
    if layerOrder == nil then
      layerOrder = 1
    end
    setNodeSeq(node, blitters[layerOrder].sequence)
  end
  local seq = blitters[#blitters].sequence
  for i = 1, #layer.insertedCameraQueues do
    local cameraQueue = layer.insertedCameraQueues[i]
    local newSequences = {}
    for j = 1, #cameraQueue.cameras do
      local camera = cameraQueue.cameras[j]
      local prevCamera = cameraQueue.cameras[j-1]
      -- If original sequences of adjcent cameras are equal,
      -- the new sequences should be equal as well
      -- This is to ensure product behaves the same as desktop-editor
      if prevCamera == nil or prevCamera:GetSequence() < camera:GetSequence() then
        seq = seq + 1
      end
      table.insert(newSequences, seq)
    end
    for j = 1, #cameraQueue.cameras do
      local camera = cameraQueue.cameras[j]
      camera:SetSequence(newSequences[j])
    end
    cameraQueue.funOnSeqChanged();
  end
end

function RenderQueue:_Update()
  local blitterIndex = #self.blitters
  local activeLayers = {}
  local layerText = ""
  local seqText = ""
  for layerIndex = #self.layers, 1, -1 do
    local layer = self.layers[layerIndex]
    local blitter = self.blitters[blitterIndex]
    if layer.active then
      table.insert(activeLayers, 1, layer)
      layer.blitter = blitter
      layer.blitter.active = true
      for k = 1, layer.blitterCount - 1 do
        layer.blitter = layer.blitter.prev
        layer.blitter.active = true
      end
      blitterIndex = blitterIndex - layer.blitterCount
      layerText = string.format(
        "%s(%d-%d) %s",
        layer.id,
        layer.blitterCount,
        layer.insertedCameraCount,
        layerText
      )
      seqText = string.format(
        "%d %s",
        layer.blitter.sequence,
        seqText
      )
    else
      layer.blitter = nil
    end
    while #layer.cameras > layer.blitterCount do
      local camera = layer.cameras[1]
      table.remove(layer.cameras, 1)
      self:_ReleaseCamera(camera)
    end
    while #layer.cameras < layer.blitterCount do
      local camera = nil
      camera = self:_ObtainCamera()
      table.insert(layer.cameras, 1, camera)
    end
  end
  while blitterIndex >= 1 do
    self.blitters[blitterIndex].active = false
    blitterIndex = blitterIndex - 1
  end
  for blitterIndex = #self.blitters, 1, -1 do
    local blitter = self.blitters[blitterIndex]
    if blitter.active then
      blitter.render:SetShow(true)
    else
      blitter.render:SetShow(false)
    end
  end
  -- to avoid node conflict with video detect input
  if #activeLayers > 0 then
    activeLayers[1].blitter.render:SetShow(false)
  end
  for layerIndex = 1, #activeLayers do
    local layer = activeLayers[layerIndex]
    local nextLayer = activeLayers[layerIndex + 1]
    self:_CheckLayerNodes(layer)
    self:_ReLinkLayer(layer, nextLayer)
    self:_ReSeqLayer(layer)
  end
  WARNING("SUNTYLOG: Active Layers: " .. layerText)
  WARNING("SUNTYLOG: Active Seqs: " .. seqText)
end



--layerID: id of the layer after which the cameras are inserting.
--cameras: the inserting cameras
--funcSettingInputTexture: to set the input texture of the cameras.
--  funcSettingInputTexture(texture) -> void
--funcGettingOutputTexture: to get the output texture of the cameras.
--  funcGettingOutputTexture(void) -> texture
--funcOnSequenceChanged: to notify the camera sequence is changed
--return: handle of the cameras, nil for failure
function RenderQueue:InsertCameras(
  layerID, 
  cameras, 
  funcSettingInputTexture, 
  funcGettingOutputTexture,
  funcOnSequenceChanged)
  local layer, order = self:_FindLayer(layerID)
  if layer == nil or order ~= nil then
    LOG("RenderQueue:InsertCameras failed: invalid layer: " .. tostring(layerID))
    return nil
  end
  if not layer.active then
    LOG("RenderQueue:InsertCameras failed: inactive layer: " .. tostring(layerID))
    return nil
  end
  local nextLayer = self:_GetNextActiveLayer(layer.index)
  if nextLayer == nil then
    LOG("RenderQueue:InsertCameras failed: cannot insert into the last active layer: "
      .. tostring(layerID))
    return nil
  end
  local handle = self:_AddCameraQueue(
    layer, 
    { unpack(cameras) }, 
    funcSettingInputTexture, 
    funcGettingOutputTexture,
    funcOnSequenceChanged
  )
  if handle == nil then
    LOG("RenderQueue:InsertCameras failed: incorrect cameras or funcs");
    return nil
  end
  self:_CheckLayerNodes(layer)
  self:_ReLinkLayer(layer, nextLayer)
  self:_ReSeqLayer(layer)
  return handle
end

function RenderQueue:RemoveCameras(handle)
  local layer = self:_RemoveCameraQueue(handle)
  local nextLayer = self:_GetNextActiveLayer(layer.index)
  if layer ~= nil then
    self:_CheckLayerNodes(layer)
    self:_ReLinkLayer(layer, nextLayer)
    self:_ReSeqLayer(layer)
  end
end

function RenderQueue:_AddCameraQueue(layer, cameras, fnSetTex, fnGetTex, funOnSeqChanged)
  if cameras == nil or fnSetTex == nil or fnGetTex == nil or funOnSeqChanged == nil then
    return nil
  end
  if layer.insertedCameraCount + #cameras >= defines.seqInterval then
    return nil
  end
  table.sort(
    cameras,
    function(a, b)
      return a:GetSequence() < b:GetSequence()
    end
  )
  local cameraQueue = {
    layer = layer,
    cameras = cameras,
    fnSetTex = fnSetTex,
    fnGetTex = fnGetTex,
    funOnSeqChanged = funOnSeqChanged
  }
  table.insert(layer.insertedCameraQueues, cameraQueue)
  layer.insertedCameraCount = layer.insertedCameraCount + #cameras
  return cameraQueue
end

function RenderQueue:_RemoveCameraQueue(cameraQueue)
  if cameraQueue == nil then
    return nil
  end
  local layer = cameraQueue.layer
  if layer == nil then
    return nil
  end
  local cameraQueueIndex = 0
  for i = 1, #layer.insertedCameraQueues do
    if layer.insertedCameraQueues[i] == cameraQueue then
      cameraQueueIndex = i
      break
    end
  end
  if cameraQueueIndex == 0 then
    return nil
  end
  table.remove(layer.insertedCameraQueues, cameraQueueIndex)
  layer.insertedCameraCount = layer.insertedCameraCount - #cameraQueue.cameras
  return layer
end

function RenderQueue:AddObjs(obj)
  table.insert(self.subLayerObjs, obj)
end

function RenderQueue:CommitSubLayerObjects(layerID)
  local layer, order = self:_FindLayer(layerID)
  if layer == nil or order ~= nil then
    return false
  end
  for k, node in pairs(layer.nodes) do
    if node.GetRenderOrder ~= nil then
      table.insert(self.subLayerObjs, node)
    end
  end
  if #self.subLayerObjs == 0 then
    return true
  end
  table.sort(
    self.subLayerObjs,
    function (a, b)
      return (a:GetRenderOrder()) < (b:GetRenderOrder())
    end
  )
  local lastRenderOrder = nil
  local layerOrder = 1
  --local renderOrderText = ""
  for i = 1, #self.subLayerObjs do
    local obj = self.subLayerObjs[i]
    local currRenderOrder = obj:GetRenderOrder()
    --renderOrderText = renderOrderText .. " " .. tostring(currRenderOrder)
    if lastRenderOrder ~= nil and currRenderOrder - lastRenderOrder > 0.000001 then
      layerOrder = layerOrder + 1
    end
    obj.layerOrder = layerOrder
    obj.renderLayer = string.format("%s-%d", layerID, layerOrder)
    lastRenderOrder = currRenderOrder
    layer.nodes[getNodeID(obj)] = obj
  end
  layer.blitterCount = layerOrder
  layer.active = true
  self:_Update()
  self.subLayerObjs = {}
  --LOG("SUNTYLOG: SubObj Orders: " .. renderOrderText)
  return true
end

function RenderQueue:SetMaxFaceOrder(maxz)
  self.facemaxz = maxz;
end

function RenderQueue:GetMaxFaceOrder()
  return self.facemaxz;
end

function RenderQueue:TestInsertCameras()
  local mainCamera = self.cameras[1]
  local cameras = {}
  local textures = {}
  local renders = {}
  local seq = -10
  local count = 3
  local near = mainCamera:GetNear()
  local far = mainCamera:GetFar()
  local pos = mainCamera:GetWorldPosition()
  local look = pos + mainCamera:GetForward()
  local up = mainCamera:GetUp()
  for i = 1, count do
    local c = self:_CreateCamera(i, near, far, pos, look, up)
    local rt, tx = self:_CreateBlitterRT(ae.RenderTargetEntity.ST_SWAP_UNIQUE)
    local r = self:_CreateBlitterRender(seq)
    c:Activate()
    c:SetSequence(seq)
    c:AttachRenderTarget(rt)
    r:SetShow(true)
    table.insert(cameras, c)
    table.insert(textures, tx)
    table.insert(renders, r)
    seq = seq + 1
  end
  for i = 1, count - 1 do
    renders[i+1]:SetParameter(
      ae.ShaderEntity.TEXTURE_DIFFUSE,
      textures[i]
    )
  end
  local setter = function(tex)
    renders[1]:SetParameter(
      ae.ShaderEntity.TEXTURE_DIFFUSE,
      tex
    )
  end
  local getter = function()
    return textures[#textures]
  end
  local onSeq = function()
    LOG("SUNTYLOG: onSeq")
    for i = 1, #renders do
      renders[i]:SetSequence(cameras[i]:GetSequence())
    end
  end
  local handle = self:InsertCameras(
    "Input",
    cameras,
    setter,
    getter,
    onSeq
  )
  self.testInsertHandle = handle
  if handle == nil then
    LOG("SUNTYLOG: InsertCameras failed")
  else
    LOG("SUNTYLOG: InsertCameras succeed")
    self:_Update()
  end
end

function RenderQueue:SetOutputRenderTarget(renderTarget, texture)
  local lastBlitter = self.blitters[#self.blitters]
  lastBlitter.renderTarget = renderTarget
  lastBlitter.texture = texture
  self:_Update()
end

return RenderQueue