local libvenuscore = require "libvenuscore"
local mathfunction = require "mathfunction"
local Types = require "venuscore.rtti.types"

local BlueUtility = require "bluecore.blueutility"
local BD = require "bluecore.bluedefined"
local AE = require "apolloengine"

local BlueNode = libvenuscore.Object:extend("BlueNode");

BlueNode.nodeName = "Default"
BlueNode.nodeType = BD.MEMBER_FUNCTION_NODE
BlueNode.eventType = BD.EventType.Normal;
BlueNode.needDelay = false;
BlueNode.ReflectFilterFtn = nil; --反射过滤函数  需要反射节点信息的子类node重写

BlueNode.objectExtend = BlueNode.extend; --Object的extend函数地址

BlueNode:MemberRegister("uid")
BlueNode:MemberRegister(BD.LUA_CLASS_PATH)
BlueNode:MemberRegister("pos")
BlueNode:MemberRegister("inputs")
BlueNode:MemberRegister("outputs")
BlueNode:MemberRegister("execInputs")
BlueNode:MemberRegister("execOutputs")
BlueNode:MemberRegister("debugStatus")


function BlueNode:extend(...)
	local clazz = BlueNode.objectExtend(self, ...)
	clazz.infoTable = {}
	clazz.infoTable[BD.PIN_DATA_INPUT] = {}
	clazz.infoTable[BD.PIN_DATA_OUTPUT] = {}
	clazz.infoTable[BD.PIN_EXEC_INPUT] = {}
	clazz.infoTable[BD.PIN_EXEC_OUTPUT] = {}

	return clazz ;
end

-- 运行模式下接口
function BlueNode:new(graph)

	self.graph = graph

	-- 运行模式下数据
	self.dependInputs = {}
	self.nextExecNodes = {}
	self.nextExecIndex = {}
	self.nextExecNum = 0
	self.inputArgs = {};
	self.outputResults = {};

	---编辑模式下用的数据
	self.uid = graph and graph:_NextID() or -1;
	self[BD.LUA_CLASS_PATH] = nil
	self.pos = mathfunction.vector2(0.0,0.0)

	-- 编译与调试状态
	self.debugStatus = false;
	self.compileResult = nil; --  自检编译结果
	self.checkResult = nil; -- blueprint编译结果
	self.errorWarning = nil;
	self.modifyMessage = nil;
  	self.isNeedUpdate = true; --输入依赖的执行节点是否需要调用Update

	self.inputs = {}
	self.outputs = {}
	self.execInputs = {}
	self.execOutputs = {}

	--允许运行(事件节点专用)
	--self.allowRunning = true;
	--self.runningonce = false;

end

--检查节点是否允许克隆
function BlueNode:IsAllowDuplicate()
	return self:Compile();
	-- body
end

--是否允许删除
function BlueNode:IsAllowDelete()
	return true;
	-- body
end


--节点克隆 
--并支持extend bluenode节点的部分字面值类型复制
function BlueNode:Duplicate(pos)

  local bluePrintNode = self:_OnDuplicate();
  bluePrintNode.pos = pos;
  if self:GetTypeName() == "BlueNode" then
  	for k,v in pairs(bluePrintNode.inputs) do
  		local pinInfoRtti = self.infoTable[BD.PIN_DATA_INPUT][k].rtti;
  		if not self:NotCopyRttiType(pinInfoRtti) then
  			bluePrintNode.inputs[k].literal = self.inputs[k].literal;
  		end
  	end
  end
  self.graph:InsertNodeInfo(bluePrintNode);
  return bluePrintNode;

end

--暂时不支持拷贝的rtti类型 涉及到资源依赖的game/scene同步
function BlueNode:NotCopyRttiType(rtti) 

	local tbl = BlueUtility.notSupportRttiCopyList;
	for _,v in pairs(tbl) do
		if rtti:isTypeOrDriverType(v) then
		 	return true;
		end
	end
	return false;

end

--子类根据实际构造函数需要的参数复写调用对应的new函数
function BlueNode:_OnDuplicate()
	local luaPath = self[BD.LUA_CLASS_PATH];
  local clazz = require(luaPath); --setvariable的luapath为tbl  其他一般都是路径字符串
  local bluePrintNode = clazz(self.graph); --new实例
  bluePrintNode[BD.LUA_CLASS_PATH] = luaPath;
	bluePrintNode:EditorCreate();
  return bluePrintNode;
end


-- 派生类实现
-- 作用: 运行时每帧调用
function BlueNode:_OnUpdate(args)
end

-- 派生类实现(delay节点)
-- 作用: 停止运行时候回调(比如释放素材,编辑器停止运行)
function BlueNode:_OnStop()
end

-- 派生类实现(delay节点)
-- 作用: 重置场景时候回调(比如音乐会stop和start)
function BlueNode:_OnReset()
end

-- 派生类实现
-- 作用: 编译时回调
-- 时机: 输入依赖建立前回调
-- 须知: 编译失败 内部设置errorString(作为提示) 并且返回false(中断编译过程)
function BlueNode:_OnCompile()
	return true
end

function BlueNode:AppendCompileResult(appendStr)
	self.checkResult = self.checkResult and string.format("%s\n%s", self.checkResult, appendStr) or appendStr
	ERROR(string.format("[Compile] Node[%s %s]:%s", tostring(self.uid), tostring(self.nodeName), self.checkResult));
end

function BlueNode:Compile()
	if not self:_OnCompile() then
		ERROR(string.format( "Compile Error\nCompile Result:%s\nCheck Result:%s",
			tostring(self.compileResult),
			tostring(self.checkResult)));
		return false
	end
	if _KRATOSEDITOR then
		self:ClearModifyMessage();
	end
  if next(self.execInputs) then --有执行输入引脚且是连接状态 则不执行Update
    --FIXME(hjh) 要获取的输入依赖来源于执行节点但执行输入没有连线 venus执行一次update  Unreal会报Warning
    for _,v in pairs(self.execInputs) do
      if next(v.links) then
        self.isNeedUpdate = false;
      end
    end
    --类似getvariable/普通加减乘除计算节点需要调用update...
  end
	return true
end

-- 当引脚的数据类型和实际输入类型不一样时候 允许具体的BlueNode根据自己的情况判断
-- pin 当前引脚的pin
-- pinInfo 当前引脚的info
-- value 输入的值(比如字面值)
function BlueNode:LiteralMismatchOnCompile(pin, pinInfo)
	return false
end


function BlueNode:_OnEditorChanged()
	-- body
end

function BlueNode:EditorChanged() --信息修改
	self:_OnEditorChanged();
	-- body
end

-- 节点创建前处理
-- 当节点在蓝图上(交互)创建时候回调,反序列化不回调
-- 在inputs/outputs创建之前 主要用于修改InfoTable
function BlueNode:_OnBeginCreateEd()

end

-- 节点创建后处理
-- 当节点在蓝图上(交互)创建时候回调,反序列化不回调
-- 当构造函数和反序列化有冲突,可以把原构造部分移到这里
-- 到这里已完成inputs/outputs/execinputs/execoutputs的创建
function BlueNode:_OnCreateWithEditor()

end

-- 增加修改提示,只用在编辑器,蓝图中代码自动完成的操作需提示用户,编译后消除
function BlueNode:AppendModifyMessage(msg)
	if _KRATOSEDITOR then
		self.modifyMessage = self.modifyMessage and self.modifyMessage .. "\n" .. msg or msg
	end
end

-- 目前如果没有链接的话,会一直提示,除非重新加载
function BlueNode:ClearModifyMessage()
	self.modifyMessage = nil
end

function BlueNode:EditorCreate(...)

	self:_OnBeginCreateEd(...);

	local graph = self.graph

	self.inputs = {}
	self.outputs = {}
	self.execInputs = {}
	self.execOutputs = {}

	for _, input in pairs(self.infoTable[BD.PIN_DATA_INPUT]) do
		local literal = BlueUtility:GenerateLiteral(input.rtti, input.default)  -- 字面量
		local inputPin = { uid = graph and graph:_NextID() or nil, nodeUid = self.uid, pinType = BD.PIN_DATA_INPUT, argId = input.argId, links = {},
											 useLiteral = true, literal = literal}
		table.insert(self.inputs, inputPin)
		WARNING("inputPin uid = "..tostring(inputPin.uid))
	end

	for _, output in pairs(self.infoTable[BD.PIN_DATA_OUTPUT]) do
		local outputPin = { uid = graph and graph:_NextID() or nil, nodeUid = self.uid, pinType = BD.PIN_DATA_OUTPUT, argId = output.argId, links = {}}
		table.insert(self.outputs, outputPin)
		WARNING("outputPin uid = "..tostring(outputPin.uid))
	end

	for _, execInput in pairs(self.infoTable[BD.PIN_EXEC_INPUT]) do
		local execPin = { uid = graph and graph:_NextID() or nil, nodeUid = self.uid, pinType = BD.PIN_EXEC_INPUT, argId = execInput.argId, links = {}}
		table.insert(self.execInputs, execPin)
		WARNING("execPin uid = "..tostring(execPin.uid))
	end

	for _, execOutput in pairs(self.infoTable[BD.PIN_EXEC_OUTPUT]) do
		local execPin = { uid = graph and graph:_NextID() or nil, nodeUid = self.uid, pinType = BD.PIN_EXEC_OUTPUT, argId = execOutput.argId, links = {}}
		table.insert(self.execOutputs, execPin)
		WARNING("execPin uid = "..tostring(execPin.uid))
	end


	self:_OnCreateWithEditor(...);
end


function BlueNode:GetOutputByIndex(index)
	-- 派生类实现接口
	-- 获取这个节点的输出
	return self.outputResults[index];
end

function BlueNode:SetDependencyInputsNode(argIdx, instance, outputIdx)
  if _KRATOSEDITOR and self.dependInputs[argIdx] ~= nil then
    WARNING("node(id:"..tostring(self.uid)..",name:"..tostring(self.nodeName)..") input "..tostring(argIdx).." changed")
  end -- 如果是在一次编译中打印这个就是有问题,但是不同一次编译打印这个代表修改了输入依赖
	self.dependInputs[argIdx] = { instance, outputIdx }
end

function BlueNode:_SkipUpdateAsInput()
  if self.nodeType == BD.EVENT_FUNCTION_NODE or self.nodeType == BD.CUSTOM_EVENT_NODE then
    return true  -- undefined behaviour  --FIXME(hjh)事件节点没有执行 但需要获取输出值 此时返回nil(hack)
  elseif self.nextExecNum ~= 0 then
    --LOG("exec node, do NOT update when input dependency")
    return true
  elseif not self.isNeedUpdate then
    return true
  end
  return false
end

-- 参数 exec_idx : = nil 代表由于输入依赖导致的update调用 ; 非nil的情况 节点根据参数和自身情况进行调用

function BlueNode:Update(execIdx)

	-- Step.0 判断是否是输入依赖而导致的update调用,但节点有执行输出,这种情况不实际执行
	--       e.g 对于if else 如果执行输出输入都没有连接,那么update每次都会被抵用
	if execIdx == nil and self:_SkipUpdateAsInput() then
	  return
  end

	-- Step.1 先执行输入的依赖节点update 并获取参数
	self:_GetInput();

	-- 可插入断点
 	if _KRATOSEDITOR then
    local BuEd = require "bluecore.editor.blueutility_ed"
    BuEd:BlueDebugYield(self);
    --ERROR("BlueNode Debug");
	end

	-- Step.2 执行本方法
	--        e.g 对于 for each节点,重写Update方法; 对于 for loop break多执行输入节点,需要判断exec_idx
	self.outputResults = {self:_OnUpdate(self.inputArgs)}; -- 每个节点执行完毕返回下一个需要执行的节点
	-- {1,"abc",nil,true} is ok , args[3] is nil args[4] is true , but NOT for ipairs

	-- Step.3 返回输出执行引脚序号
	return  self:_OnNextBranch() ;
end


function BlueNode:_GetInput()
	self.inputArgs = {}
	for argIdx, input in pairs(self.dependInputs) do
		input[1]:Update();  -- 不处理返回的节点
		self.inputArgs[argIdx] = input[1]:GetOutputByIndex(input[2]);
	end
	return self.inputArgs
end


function BlueNode:Stop()
	self.errorWarning = nil;
	self.inputArgs = {}; -- 释放所有作为参数传入的对象引用
	self.outputResults = {};
	self:_OnStop();
end


function BlueNode:Reset()
	self:_OnReset();				-- 先执行Reset,这样节点还可以使用inputArgs
	self.errorWarning = nil;
	self.inputArgs = {};
	self.outputResults = {};
end


-- 运行时输出执行引脚索引
-- BLueNode.Update()  BluePrint.<EventFun>() 仅在两处调用
-- 当派生类重新_onUpdate并且有多个执行输出引脚时，需要重写_OnNextBranch
-- 比如if else 等有两个或多个执行输出的,需要派生类重载
-- 返回 nil,负数,0 不执行后续蓝图节点
function BlueNode:_OnNextBranch()
	return 1
end


function BlueNode:setExecOutput(index , nextNode, nextNodeExecPin)
	self.nextExecNodes[index] = nextNode;
	self.nextExecIndex[index] = nextNodeExecPin;
	self.nextExecNum = self.nextExecNum + 1 ;
end

function BlueNode:_CheckPin(pinType, argId, name)
	if _KRATOSEDITOR then
		local last = self.infoTable[pinType][argId]
		if last then
			ERROR("pin "..tostring(argId)..", type ".. tostring(pinType).." has been occupied, last:".. last.name .. " current:".. name);
			assert(not last , "pin ArgId has been occupied");
		end

		if argId > 1 then
			last = self.infoTable[pinType][argId - 1]
			if last == nil then
				ERROR("pin "..tostring(argId)..", type ".. tostring(pinType).." skip "..tostring(argId - 1));
				assert( last ~= nil , "pin ArgId skip");
			end
		end
	end
end

--每一个register都是存了引脚的重要信息
--- 编辑器模式下接口  (在蓝图中右键Create Node的时候 选中SetLocalPosition节点 将会调用如下方法 生成节点/和引脚)
function BlueNode:RegisterInput(argId, rtti, name, tips, default, delAble, groupName)

	self:_CheckPin(BD.PIN_DATA_INPUT, argId, name);

	local input = { rtti = rtti, argId = argId, name = name, tips = tips, default = default,
									delAble = delAble, groupName = groupName, pinType = BD.PIN_DATA_INPUT};
	self.infoTable[BD.PIN_DATA_INPUT][argId] = input
end


function BlueNode:RegisterOutput(argId, rtti, name, tips, default, delAble, groupName)

	self:_CheckPin(BD.PIN_DATA_OUTPUT, argId, name);

	-- 输出不用设置默认值或者字面值
	local output = { rtti = rtti, argId = argId, name = name, tips = tips, default = nil,
									 delAble = delAble, groupName = groupName, pinType = BD.PIN_DATA_OUTPUT };
	self.infoTable[BD.PIN_DATA_OUTPUT][argId] = output
end

-- 如果引脚可变的节点类 会移除原来旧的引脚 参考 setvariable.lua
-- 如果引脚数目可变化的节点(实例) 参考 make_array.lua
function BlueNode:UnRegisterInput(argId)
	self.infoTable[BD.PIN_DATA_INPUT][argId] = nil
end

function BlueNode:UnRegisterOutput(argId)
	self.infoTable[BD.PIN_DATA_OUTPUT][argId] = nil
end


function BlueNode:RegisterExecInput(argId, name, tips, abort, delAble)

	self:_CheckPin(BD.PIN_EXEC_INPUT, argId, name);

	local info = { rtti = Types.ExecPinType, argId = argId, name = name, tips = tips, abort = abort,
								 delAble = delAble or false,
								 pinType = BD.PIN_EXEC_INPUT}
	self.infoTable[BD.PIN_EXEC_INPUT][argId] = info

end

function BlueNode:RegisterExecOutput(argId, name, tips, delAble)

	self:_CheckPin(BD.PIN_EXEC_OUTPUT, argId, name);

	local execoutput = { rtti = Types.ExecPinType, argId = argId, name = name, tips = tips, abort = false,
											 delAble = delAble or false,
											 pinType = BD.PIN_EXEC_OUTPUT}
	self.infoTable[BD.PIN_EXEC_OUTPUT][argId] = execoutput

end


function BlueNode:RegisterMoreInput(name, addName, addTips, onAdd, delName, delTips, onDel)
	local bluePrintPinAdjustInfo = {
		name = name,
		add = {
			opname = addName,
			tips = addTips,
			onAdd = onAdd
		},
	  del = {
			opname = delName,
			tips = delTips,
			onDel = onDel,
		}};
	self.adjustTable = self.adjustTable or {}
	self.adjustTable[BD.MORE_INPUT] = bluePrintPinAdjustInfo
end

function BlueNode:RegisterMoreOutput(name, addName, addTips, onAdd, delName, delTips, onDel)
	local BluePrintPinAdjustInfo = {
		name = name,
		add = {
			opname = addName,
			tips = addTips,
			onAdd = onAdd
		},
		del = {
			opname = delName,
			tips = delTips,
			onDel = onDel,
		}};
	self.adjustTable = self.adjustTable or {}
	self.adjustTable[BD.MORE_OUTPUT] = BluePrintPinAdjustInfo
end

function BlueNode:RegisterMoreExecInput(name, addName, addTips, onAdd, delName, delTips, onDel)
	local BluePrintPinAdjustInfo = {
		name = name,
		add = {
			opname = addName,
			tips = addTips,
			onAdd = onAdd
		},
		del = {
			opname = delName,
			tips = delTips,
			onDel = onDel,
		}};
	self.adjustTable = self.adjustTable or {}
	self.adjustTable[BD.MORE_EXEC_INPUT] = BluePrintPinAdjustInfo
end

function BlueNode:RegisterMoreExecOutput(name, addName, addTips, onAdd, delName, delTips, onDel)
	local BluePrintPinAdjustInfo = {
		name = name,
		add = {
			opname = addName,
			tips = addTips,
			onAdd = onAdd
		},
		del = {
			opname = delName,
			tips = delTips,
			onDel = onDel,
		}};
	self.adjustTable = self.adjustTable or {}
	self.adjustTable[BD.MORE_EXEC_OUTPUT] = BluePrintPinAdjustInfo
end

-- 节点需要每帧通知 注册回调
function BlueNode:RegisterOnTick(onTickFunc)
	self._OnTickCallBack = onTickFunc
end

function BlueNode:GetInputsType()
    return self.infoTable[BD.PIN_DATA_INPUT];
end

function BlueNode:GetOutputsType()
	return self.infoTable[BD.PIN_DATA_OUTPUT];
end

function BlueNode:GetExecInput()
	return self.infoTable[BD.PIN_EXEC_INPUT];
end

function BlueNode:GetExecOutput()
	return self.infoTable[BD.PIN_EXEC_OUTPUT];
end

function BlueNode:SetFunctionName(Name)
	self.nodeName = Name
end

function BlueNode:GetUid()
	return self.uid
end

function BlueNode:GetName()
	return self.nodeName
end

function BlueNode:GetGraph()
	return self.graph ; -- maybe blueprint/bluefunction
end

function BlueNode:SetFunctionType(nodeType)
	self.nodeType = nodeType
end

function BlueNode:GetFunctionType()
	return self.nodeType
end

function BlueNode:IsNodeType(rhsType)
	return self.nodeType == rhsType
end

function BlueNode:SetEventType(eventType)
  self.eventType = eventType;
end

function BlueNode:GetEventType()
  return self.eventType;
end

function BlueNode:CanCreateMoreInput()
	if not self.adjustTable then
		return false;
	end
	return self.adjustTable[BD.MORE_INPUT] ~= nil
end

function BlueNode:CanCreateMoreOutput()
	if not self.adjustTable then
		return false;
	end
	return self.adjustTable[BD.MORE_OUTPUT] ~= nil
end

function BlueNode:CanCreateMoreExecInput()
	if not self.adjustTable then
		return false;
	end
	return self.adjustTable[BD.MORE_EXEC_INPUT] ~= nil
end

function BlueNode:CanCreateMoreExecOutput()
	if not self.adjustTable then
		return false;
	end
	return self.adjustTable[BD.MORE_EXEC_OUTPUT] ~= nil
end

function BlueNode:GetMoreInputsMethod()
	return self.adjustTable[BD.MORE_INPUT]
end

function BlueNode:GetMoreOutputsMethod()
	return self.adjustTable[BD.MORE_OUTPUT]
end

function BlueNode:GetMoreExecInputsMethod()
	return self.adjustTable[BD.MORE_EXEC_INPUT]
end

function BlueNode:GetMoreExecOutputsMethod()
	return self.adjustTable[BD.MORE_EXEC_OUTPUT]
end

function BlueNode:_OnDeserialize()

end

function BlueNode:_OnDeserializePost()

end

function BlueNode:_OnLink(selfPin, otherNode, otherPin)

end

function BlueNode:_OnUnLink(unlinkPin)

end

-- 断开完成-任何一次断开都会调用-基类暂时不判断是否完全断开
function BlueNode:unLinked(unlinkPin)
	self:_OnUnLink(unlinkPin)
end

--如果某节点需要改造，则需要新写一个节点，老节点增加该函数设为true就不会显示出来
function BlueNode:isHide()
	return false;
end

-- 连接完成-任何一次连接都会调用
function BlueNode:Linked(selfPin, otherNode, otherPin)
	self:_OnLink(selfPin, otherNode, otherPin)
end

function BlueNode:GetPinRtti(pin)
	return self.infoTable[pin.pinType][pin.argId].rtti
end

-- 编辑器 通过本node的pinId 找到pinInfo
function BlueNode:GetPinInfoByPin(pin)
	local pinInfo = self.infoTable[pin.pinType][pin.argId]
	assert(pinInfo, "GetPinInfoByPin "..tostring(pin.argId)..","..tostring(pin.pinType).." is not my own pin")
	return pinInfo
end

-- 编辑器 通过本node的pinId 找到pin
function BlueNode:GetPinByPinId(pinId)
	for _, input in pairs(self.inputs) do
		if input.uid == pinId then return input end
	end
	for _, output in pairs(self.outputs) do
		if output.uid == pinId then return output end
	end
	for _, execInput in pairs(self.execInputs) do
		if execInput.uid == pinId then return execInput end
	end
	for _, execOutput in pairs(self.execOutputs) do
		if execOutput.uid == pinId then return execOutput end
	end
	assert(false,"GetPinByPinId "..tostring(pinId).." is not my own pin")
end

-- 编辑器 通过本node的pinInfo 找到pin
function BlueNode:GetPinByInfo(pinInfo)
	local pin
	if pinInfo.pinType == BD.PIN_DATA_INPUT then
		pin = self.inputs[pinInfo.argId] ;
	elseif pinInfo.pinType == BD.PIN_DATA_OUTPUT then
		pin = self.outputs[pinInfo.argId] ;
	elseif pinInfo.pinType == BD.PIN_EXEC_INPUT then
		pin = self.execInputs[pinInfo.argId] ;
	elseif pinInfo.pinType == BD.PIN_EXEC_OUTPUT then
		pin = self.execOutputs[pinInfo.argId] ;
	else
		assert(false,"GetPinByInfo pinType not support "..tostring(pinInfo.pinType));
	end
	assert(pin, "GetPinByInfo pin not found");
	return pin
end


function BlueNode:GetPinInfoByPinId(pinId)
	local pin = self:GetPinByPinId(pinId)
	return self:GetPinInfoByPin(pin);
end

function BlueNode:SetPinRtti(pin, rtti)
	local info = self.infoTable[pin.pinType][pin.argId]
	if info.rtti == Types.AnyType or info.rtti == Types.AnyArrayType then
		info.rtti = rtti
		if self._AdjustRtti then
			self:_AdjustRtti();
		end
	--else
	--	ERROR("[SetPinRtti] rtti is not any")
	end
end

-- 输入引脚是否支持数组(即使rtti不是数组的情况下)
function BlueNode:IsInputArrayCompatible(pinInfo)
	return false
end

-- 编辑器接口 WITH_EDITOR
if _KRATOSEDITOR then

	function BlueNode:_OnUpdateByEditor()
		-- 派生类实现
	end

	-- 每次渲染BlueNode的时候会调用 方便BlueNode本身更新和同步(目前用于blue_comp_ref.lua)
	-- 注意 UpdateByEditor 只会在editor scene中被调用, 单独修改editor scene中bluenode并不会同步到preview/game scene
	function BlueNode:UpdateByEditor()
		self:_OnUpdateByEditor()
	end

	-- 告知节点即将在蓝图上删除(这时,所有连线都还保留)
	function BlueNode:DeleteEd()
		self:_OnDeleteEd();
	end

	function BlueNode:_OnDeleteEd()

	end

end -- _KRATOSEDITOR


return BlueNode;





