local Object       = require "classic"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local apollonode   = require "apolloutility.apollonode"
local venuscore    = require "venuscore"
local defined      = require "apolloutility.defiend"

local HairColorBboxRender = Object:extend();

function HairColorBboxRender:new(scene, render_sequence, mask_layer, material_path, width, height)
    self.scene        = scene; 
    self.node         = self.scene:CreateNode(apolloengine.Node.CT_NODE);
    local rootnode = self.node:GetRoot();
    rootnode:DetachNode(self.node);
    self.render       = self.node:CreateComponent(apolloengine.Node.CT_RENDER);
    self.isshow = true;
    
    self.camera       = self.node:CreateComponent(apolloengine.Node.CT_CAMERA);
    self.frame_width  = width;
    self.frame_height = height;
    self.initialized  = false;

    local clear_color   = mathfunction.Color(0.0, 0.0, 0.0, 0.0);
    local fbo_size      = mathfunction.vector2(self.frame_width, self.frame_height);
    self.render_target  = apolloengine.RenderTargetEntity();
    self.render_target:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
    apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
    mathfunction.vector4(0, 0, self.frame_width, self.frame_height),
    fbo_size));

    self.image_gpu = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    self.image_gpu:PushMetadata(apolloengine.TextureRenderMetadata(
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        fbo_size));

    local depth = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_DEPTH_STENCIL);
    depth:PushMetadata(
      apolloengine.DepthRenderBufferMetadata(
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        fbo_size,
        apolloengine.TextureEntity.PF_DEPTH24_STENCIL8
      ));
    
    self.render_target:CreateResource();

    self.camera:SetSequenceCulling(true);
    self.camera:SetLayerMaskNothing();
    self.camera:AddLayerMask(mask_layer);
    self.camera:Activate();
    self.camera:SetSequence(render_sequence);
    self.camera:SetClearColor(clear_color);  
    self.camera:FixedResolution(fbo_size);
    self.camera:AttachRenderTarget(self.render_target);

    self.vertexstream = apolloengine.VertexStream();--创建顶点流
    self.vertexstream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
    apolloengine.VertexBufferEntity.DT_FLOAT,
    apolloengine.VertexBufferEntity.DT_HALF_FLOAT,
    4);
    self.vertexstream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
		apolloengine.VertexBufferEntity.DT_FLOAT,
		apolloengine.VertexBufferEntity.DT_HALF_FLOAT,
		2);
    self.vertexstream:ReserveBuffer(8); --把每一个面拆开渲染
    self.indexstream = apolloengine.IndicesStream(); 
    self.indexstream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
    self.indexstream:ReserveBuffer(6);
    self.indexstream:PushIndicesData(0);
    self.indexstream:PushIndicesData(1);
    self.indexstream:PushIndicesData(2);
    self.indexstream:PushIndicesData(2);
    self.indexstream:PushIndicesData(1);
    self.indexstream:PushIndicesData(3);
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,mathfunction.vector4(0.0, 0.0, 0.0, 1.0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,mathfunction.vector4(0.0, 0.0, 0.0, 1.0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,mathfunction.vector4(0.0, 0.0, 0.0, 1.0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,mathfunction.vector4(0.0, 0.0, 0.0, 1.0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(0, 0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(0, 1));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(1, 0));
    self.vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(1, 1));


    local material = venuscore.IFileSystem:PathAssembly(material_path);
    ERROR("material path: "..material);
    self.node:SetLayer(mask_layer);
    self.render:PushMetadata(
    apolloengine.RenderObjectMaterialMetadata(
    apolloengine.PathMetadata(material)));

    self.render:PushMetadata(
		apolloengine.RenderObjectMeshMetadate( 
			apolloengine.RenderComponent.RM_TRIANGLES,
      apolloengine.ReferenceVertexMetadata(        
        apolloengine.VertexBufferEntity.MU_DYNAMIC,--使用可以修改的显存区域
        self.vertexstream),
      apolloengine.ReferenceIndicesMetadata(
        apolloengine.VertexBufferEntity.MU_STATIC,--使用不可修改的显存区域
        self.indexstream)));
    self.render:CreateResource();
    -- self.node:CreateResource(material, apolloengine.RenderComponent.RM_TRIANGLES, self.vertexstream, self.indexstream);
    self.render:SetSequence(render_sequence);

    if _KRATOSEDITOR then
        self.camera:SetResourceType(render_sequence);
        self.render:SetResourceType(render_sequence);
    end

    self.initialized   = true;
end

function HairColorBboxRender:SetShow(is_show)
    if self.isshow ~= is_show then
        self.isshow = is_show;
        if is_show then
          self.render:SetRenderProperty(apolloengine.RenderComponent.RP_SHOW);
        else
          self.render:EraseRenderProperty(apolloengine.RenderComponent.RP_SHOW);
        end  
    end
end

function HairColorBboxRender:PreUpdate(sequence)
    if not self.initialized then
        LOG("[HairColorBboxRender] preUpdate fail");
        return;
    end
    self.camera:SetSequence(sequence);
    self.render:SetSequence(sequence);
    self:SetShow(false);
end

function HairColorBboxRender:calculateRotateBbox(leftBottom, leftTop, rightBottom, rightTop, roll)
    local rollOrg = roll;
    local rolls = roll;
    if roll < 0 then
        rolls = -roll;
    else
        rolls = 90 - roll;
    end

    local cosR = math.cos(rolls / 180 * 3.14159266);
    local sinR = math.sin(rolls / 180 * 3.14159266);
    local width = rightBottom:x() - leftBottom:x();
    local height = leftBottom:y() - leftTop:y();

    local v1 = mathfunction.vector2(leftTop:x() + width*sinR*sinR, leftTop:y() - width*sinR*cosR);
    local v2 = mathfunction.vector2(rightTop:x() + height*sinR*cosR, rightTop:y() + height*sinR*sinR);
    local v3 = mathfunction.vector2(rightBottom:x() - width*sinR*sinR, rightBottom:y() + width*sinR*cosR);
    local v4 = mathfunction.vector2(leftBottom:x() - height*sinR*cosR, leftBottom:y() - height*sinR*sinR);

    if rollOrg < 0 then
        leftBottom = v4;
        rightTop = v2;
        leftTop = v1;
        rightBottom = v3;
    elseif rollOrg > 0 then
        leftBottom = v3;
        rightTop = v1;
        leftTop = v4;
        rightBottom = v2;
    end

    return leftBottom, leftTop, rightBottom, rightTop
end

function HairColorBboxRender:convertVector(point, matrix)
    local t = mathfunction.vector3(point:x(), point:y(), 1) * matrix;
    return mathfunction.vector2(t:x(), t:y());
end

function HairColorBboxRender:Update(mask_size, bbox, roll, scale, uv)
    if not self.initialized then
        LOG("[HairColorBboxRender] update fail");
        return false;
    end

    -- set show for node
    self:SetShow(true);

    local maskWidth = mask_size:x();
    local maskHeight = mask_size:y();
    local xmin = bbox:x();
    local ymin = bbox:y();
    local xmax = xmin + bbox:z();
    local ymax = ymin + bbox:w();
    local leftBottom = mathfunction.vector2(xmin, ymax);
    local leftTop = mathfunction.vector2(xmin, ymin);
    local rightBottom = mathfunction.vector2(xmax, ymax);
    local rightTop = mathfunction.vector2(xmax, ymin);

    leftBottom, leftTop, rightBottom, rightTop = self:calculateRotateBbox(leftBottom, leftTop, rightBottom, rightTop, roll);
    local projectMatrix = mathfunction.Matrix33(2 / maskWidth, 0, 0, 0, -2 / maskHeight, 0, -1, 1, 1);
    leftBottom = self:convertVector(leftBottom, projectMatrix);
    leftTop = self:convertVector(leftTop, projectMatrix);
    rightBottom = self:convertVector(rightBottom, projectMatrix);
    rightTop = self:convertVector(rightTop, projectMatrix);


    local p1 = mathfunction.vector2(math.max(0.5+0.5*uv:x()-0.5*scale, 0), math.max(0.5+0.5*uv:y()-0.5*scale, 0));
    local p2 = mathfunction.vector2(math.max(0.5+0.5*uv:x()-0.5*scale, 0), math.min(0.5+0.5*uv:y()+0.5*scale, 1));
    local p3 = mathfunction.vector2(math.min(0.5+0.5*uv:x()+0.5*scale, 1), math.max(0.5+0.5*uv:y()-0.5*scale, 0));
    local p4 = mathfunction.vector2(math.min(0.5+0.5*uv:x()+0.5*scale, 1), math.min(0.5+0.5*uv:y()+0.5*scale, 1));

    local index = self.vertexstream:GetAttributeIndex(apolloengine.ShaderEntity.ATTRIBUTE_POSITION);
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,1,mathfunction.vector4(leftTop:x(), leftTop:y(), 0 , 1));
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,2,mathfunction.vector4(leftBottom:x(), leftBottom:y(), 0 , 1));
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,3,mathfunction.vector4(rightTop:x(), rightTop:y(), 0 , 1));
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,4,mathfunction.vector4(rightBottom:x(), rightBottom:y(), 0 , 1));


    local index = self.vertexstream:GetAttributeIndex(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0);
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,1,p1);
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,2,p2);
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,3,p3);
    self.vertexstream:ChangeVertexDataWithAttributeFast(index,4,p4);
    
    self.vertexstream:SetReflushInterval(1, 4);
    self.render:ChangeVertexBuffer(self.vertexstream);

    return true;
end

function HairColorBboxRender:SetParameter(key, value)
    if key == nil or value == nil then
        LOG("[HairColorBboxRender] SetParameter nil "..key);
        return;
    end

    if self.initialized then
        self.render:SetParameter(key, value);
    end
end

function HairColorBboxRender:GetResultImage()
    return self.image_gpu;
end

function HairColorBboxRender:Clear()
    if self.node ~= nil then
        local scene = self.scene;
        scene:DeleteNode(self.node);
        self.camera = nil;
        self.vertexstream = nil;
        self.indexstream = nil;
        self.render = nil;
        self.render_target = nil;
        self.image_gpu = nil;
        self.node = nil;
        self.scene = nil;
    end
end

return HairColorBboxRender;
