
local venuscore = require "venuscore"
local venusjson = require "venusjson"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local ContentManager = require "apolloengine.content.contentmanager"
local AnimationSwitch = venuscore.VenusBehavior:extend("AnimationSwitch");

function AnimationSwitch:new()
    self.undoparameter = nil;
    self.curState = nil;
    self.animationInfoPath = "";
    self.animationInfo = {};
    self.onStarted = false;
end

function AnimationSwitch:_OnAwake()
    self:LoadAnimationInfo();
end

function AnimationSwitch:_OnUpdate(def)
    if self.onStarted == false then
        self:DoOnStart();
        self.onStarted = true;
    end
end

function AnimationSwitch:ConnectStates(state1, state2, paramName)
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end

    local transition1 = animationComp:CreateTransition(0, state1, state2);
    local transition2 = animationComp:CreateTransition(0, state2, state1);
    transition1:SetStaticID(ContentManager:GenerateUUID());
    transition2:SetStaticID(ContentManager:GenerateUUID());
    transition1.Duration = 0.1;
    transition2.Duration = 0.1;

    local parameter1 = animationComp:CreateParameter(0, 3);
    local parameter2 = animationComp:CreateParameter(0, 3);
    parameter1:SetStaticID(ContentManager:GenerateUUID());
    parameter2:SetStaticID(ContentManager:GenerateUUID());
    parameter1.Name = paramName..1;
    parameter2.Name = paramName..2;
    --parameter1.BoolValue = true;
    --parameter2.BoolValue = true;

    local conditions1 = transition1.Conditions;
    local conditions2 = transition2.Conditions;
    animationComp:SetParameterToCondition(0, transition1:GetStaticID(), conditions1[1]:GetStaticID(), parameter1:GetStaticID());
    animationComp:SetParameterToCondition(0, transition2:GetStaticID(), conditions2[1]:GetStaticID(), parameter2:GetStaticID());
end

function AnimationSwitch:DoOnStart()
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end

    if table.getn(animationComp.StateMachines) == 0 then
        animationComp:CreateStateMachine();
    end

    local animations = animationComp:GetAnimations();
    local state = {};
    local cnt = 1;
    for k, v in pairs(animations) do
        state[cnt] = animationComp:CreateState(0);
        state[cnt]:SetStaticID(ContentManager:GenerateUUID());
        state[cnt].Name = k;
        local animationEntity = animationComp:GetAnimationEntity(k);
        state[cnt]:SetMotion(animationEntity);
        cnt = cnt + 1;
    end

    for i = 1, #state do
        for j = i+1, #state do
            self:ConnectStates(state[i], state[j], "p"..i..j.."_");
        end
    end
end

function AnimationSwitch:IsInited()
  local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
  if animationComp == nil then
    return false;
  end
  if table.getn(animationComp.StateMachines) == 0 then
    return false;
  end
  return true;
end

function AnimationSwitch:Run()
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end
    animationComp:RunStateMachines();
end

function AnimationSwitch:Stop()
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end
    animationComp:StopStateMachines();
end

function AnimationSwitch:IsRunning()
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return false;
    else
        return animationComp:IsStateMachinesRunning();
    end
end

function AnimationSwitch:Pause(aniIdx, isPaused)
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end
    animationComp:Pause(self.animationInfo["name"][aniIdx], isPaused);
end

function AnimationSwitch:Play(aniIdx)
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp == nil then
        return;
    end
    animationComp:Play(self.animationInfo["name"][aniIdx]);
end

function AnimationSwitch:SwitchAnimationByIdx(animIdx)
    if self:IsAniAndFrameValid(animIdx) == true then
        self:SwitchAnimationByName(self.animationInfo["name"][animIdx]);
    else
        ERROR("AnimationSwitch SwitchAnimationByIdx animIdx: "..tostring(animIdx).."  is invalid");
    end
end

function AnimationSwitch:IsCurStatePlayFinish()
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    if animationComp ~= nil then
        local stateMachines = animationComp.StateMachines;
        if stateMachines[1] ~= nil then
            return stateMachines[1]:IsPlayFinish();
        end
    end

    return true;
end

function AnimationSwitch:IsAniAndFrameValid(animIdx)
    if animIdx then
        local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
        if animationComp ~= nil and self.animationInfo["name"] ~= nil then
            local animations = animationComp:GetAnimations();
            local aninames = self.animationInfo["name"];
            if animIdx > 0 and animIdx <= #aninames then
                return true;
            end
        end    
    end
    return false;
end

function AnimationSwitch:SwitchAnimationByName(nextAniName)
    local animationComp = self.Node:GetComponent(apolloengine.Node.CT_ANIMATION);
    local stateMachines = animationComp.StateMachines;
    if stateMachines==nil or #stateMachines==0 or stateMachines[1]:IsInTransition() then
      return;
    end
    
    if self.undoparameter ~= nil then
        self.undoparameter.BoolValue = false;
        self.undoparameter = nil;
    end
    
    local states = stateMachines[1].States;
    local transitions = stateMachines[1].Transitions;
    local parameters = stateMachines[1].Parameters;

    local nextStateId = nil;
    local nextState = nil;
    for i = 1, #states do
        local sname = states[i].MotionName;
        if sname == nextAniName then
            nextStateId = states[i]:GetStaticID();
            nextState = states[i];
        end
    end
    if nextState == nil then
        return;
    end
    
    if self.curState == nil then
        --animationComp:Play(nextAniName);
        self.curState = nextState;
        animationComp:SetDefaultState(0,nextState);
        return;
    end

    local curAniName = self.curState.MotionName;
    local curStateId = self.curState:GetStaticID();
    
    for k, transition in pairs(transitions) do
        local srcStateId = transition.SrcId;
        local dstStateId = transition.DstId;
        if srcStateId == curStateId and dstStateId == nextStateId then
            local conditions = transition.Conditions;
            local paramIndex = conditions[1].ParamIndex+1;
            --print("paramIndex "..parameters[paramIndex].Name);
            parameters[paramIndex].BoolValue = true;
            
            self.undoparameter = parameters[paramIndex];
            self.curState = nextState;
            --conditions[1].BoolValue = true;
        end
    end
end

function AnimationSwitch:LoadAnimationInfo()
    if self.animationInfoPath == "" then
        return;
    end
    local file = venusjson.LaodJsonFile(self.animationInfoPath);
    if file ~= nil then
        self.animationInfo = file;
    end
end

AnimationSwitch:MemberRegister(
    "animationInfoPath",
    venuscore.ScriptTypes.FilePathType(
        {"json"},
        nil,
        function(thiz, value)
            thiz.animationInfoPath = value;
            thiz:LoadAnimationInfo();
        end
    )
)

return AnimationSwitch;