//
//	Facet Shader for Specular and Texturing
//

// Various defines for enabling and disabling features at compile time.
// NORMALISE_NORMALS			- normalises surface normals
// DISCARD_ON_ALPHA				- discards pixels if the material's alpha is less than a threshold
// TEXTURING_0					- enables Texturing unit 0
// DIRECTIONAL_LIGHT_0			- enables directional lighting
// SPECULAR						- enables specular highlights
// POINT_LIGHT_0				- enables point lighting
// NORMAL_MAPPING				- enabled normal (bump) mapping
// PARALLAX_MAPPING 

uniform mediump vec4 INPUT_SpecularColour;
uniform highp float INPUT_Shininess;  // Must be highp else device and simulator show different results
uniform sampler2D SAMPLER_Texture0;
uniform sampler2D SAMPLER_Texture1;
uniform lowp int INPUT_Texture0Enabled;

uniform lowp int INPUT_NormalMapTextureEnabled;		// Bump mapping
uniform sampler2D SAMPLER_NormalMap;

// 2D sampler containing cartesian to polar conversion data
// The input references go from 0.0 to 1.0 in x and y which
// are equivalent to -1 to +1 centered at the middle of the 2D array.
// The output values holds distance in x and angle in y
// the angle given as 0.0 to 1.0 is equivalent
// to -PI to +PI so do a (y*2)-1 conversion.
uniform sampler2D SAMPLER_CartesianToPolar;		 

uniform int INPUT_DirectionalLight0Enabled;
uniform mediump vec3 INPUT_DirectionalLight0Direction;
uniform mediump vec4 INPUT_DirectionalLight0Specular;

uniform int INPUT_PointSpotLight0Enabled;
uniform mediump vec4 INPUT_PointSpotLight0Specular;

uniform highp float INPUT_Time;

// Parameters passed from the Vertex Shader

varying mediump vec4 VtoF_Ambient;
varying mediump vec3 VtoF_Normal;
varying mediump vec4 VtoF_Colour;						// Colour is (vertex colour * diffuse)
varying mediump vec2 VtoF_TextureCoord;
varying mediump vec2 VtoF_TextureCoordCentered;

#if (defined TEXTURING_1_SRCA_1MINUSSRCA) || (defined TEXTURING_1_1_SRCA) || (defined TEXTURING_1_CROSSFADE)
varying mediump vec2 VtoF_SecondaryTextureCoord;
#endif 

varying mediump vec3 VtoF_ViewDirection;
varying mediump vec3 VtoF_UntransformedViewDirection;

varying highp float VtoF_Time;

void main()
{
	mediump vec3 n;
	mediump float NdotL,NdotHV;           
	mediump vec4 AccumulatedColour;
	mediump vec4 MaterialColour;
	mediump float Alpha;
	mediump vec4 DirectionalLight0Effect;
	mediump vec4 PointSpotLight0Effect;
	mediump vec4 HighTemp;
	mediump vec2 newTexCoord;

	AccumulatedColour = VtoF_Ambient;

//================================================================================

	// Convert to a polar coordinate space
	mediump vec2 polarc;
	
	// High-resolution reference version using atan and sqrt:
//	polarc.y = atan(VtoF_TextureCoord.x, VtoF_TextureCoord.y) / 3.1415926;
//	polarc.x = length(VtoF_TextureCoord);

	mediump vec4 polar = texture2D(SAMPLER_CartesianToPolar,VtoF_TextureCoordCentered);
	polarc.x = length(VtoF_TextureCoord);//Centered);
	
	// Try and make use of built in multiple-accumulate.... doesn't make any difference in practice
	// polarc.x will be the length
	// polarc.y will be the angle.
//	highp vec2 polarc = vec2(polar.x, polar.z) * vec2(1.0/256.0, 1.0/256.0) + vec2(polar.y, polar.w);
	polarc.y = polar.z * (1.0/256.0) + polar.w;
	
	newTexCoord = vec2(VtoF_Time + 0.8/polarc.x, (polarc.y * 2.0)-1.0);

//================================================================================

#ifdef TEXTURING_0	
//   	MaterialColour = (VtoF_Colour * texture2D(SAMPLER_Texture0,newTexCoord)) * polarc.x;	// Fade to black
//   	MaterialColour = (VtoF_Colour * texture2D(SAMPLER_Texture0,newTexCoord)) / (1.1*r);   // Fade to white
	mediump float factor = max(1.1 * polarc.x, 0.5);
   	MaterialColour = (VtoF_Colour * texture2D(SAMPLER_Texture0,newTexCoord)) / factor;   // Fade to white
//   	MaterialColour = (VtoF_Colour * texture2D(SAMPLER_CartesianToPolar,newTexCoord));   // Fade to white
#else
	MaterialColour = VtoF_Colour;
#endif    

//================================================================================

#ifdef TEXTURING_1_SRCA_1MINUSSRCA	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour.xyz = mix(MaterialColour.xyz, Material2Colour.xyz, Material2Colour.w);
#endif    

#ifdef TEXTURING_1_1_SRCA	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour = MaterialColour + (Material2Colour * Material2Colour.w);
#endif    

#ifdef TEXTURING_1_CROSSFADE	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour.xyz = mix(MaterialColour.xyz, Material2Colour.xyz, 0.5);//INPUT_Texture1CrossFadeFactor);
#endif    

//================================================================================
    
#ifdef DISCARD_ON_ALPHA    
    if (MaterialColour.a<0.02)		// Arbitrary cut off....  
    {
    	// Discard pixel if it is considered transparent enough to be not there at all;
    	// Importantly, we don't want to write to the depth buffer...
    	// This is important for textures with characters on to appear correctly invisible
    	// around the edges when rendered against other textures...
    	discard;					
    }
#endif    

//================================================================================

	AccumulatedColour.xyz += MaterialColour.xyz;
   
   	AccumulatedColour.a = MaterialColour.a;		// Preserve alpha channel - derived from diffuse and texture
	gl_FragColor = AccumulatedColour;
}
