//================================================================================
//================================================================================
//
//	Vertex Shader for Specular and Texturing
//
//================================================================================
//================================================================================

#if (defined POINT_LIGHT_0) || (defined DIRECTIONAL_LIGHT_0) || (defined POINT_LIGHT_1)
#define LIGHTING_ENABLED
#endif

attribute vec4 ATTRIB_Vertex;
attribute vec3 ATTRIB_Normals;
attribute vec2 ATTRIB_TextureUV;
attribute vec4 ATTRIB_Colour;
attribute vec4 ATTRIB_NormalMap;
attribute vec4 ATTRIB_Tangents;
attribute vec2 ATTRIB_SecondaryTextureUV;			// Second multi-texturing UV data set.

uniform mat4 INPUT_ModelMatrix;				// Model matrix
uniform mat4 INPUT_ModelProjMatrix;			// Projection * Model matrix
uniform mat4 INPUT_NormalMatrix;

uniform vec4 INPUT_DiffuseColour;
uniform vec4 INPUT_AmbientColour;

uniform int  INPUT_DirectionalLight0Enabled;
uniform vec3 INPUT_DirectionalLight0Direction;
uniform vec3 INPUT_DirectionalLight0HalfVector;

uniform int  INPUT_PointSpotLight0Enabled;
uniform vec3 INPUT_PointSpotLight0Position;				// Position is in transformed view space...
uniform vec3 INPUT_PointSpotLight0Direction;
uniform vec3 INPUT_PointSpotLight0HalfVector;

uniform sampler2D SAMPLER_NormalMap;

// Data passed from the Vertex shader to the Facet shader:

#ifdef LIGHTING_ENABLED
varying mediump vec3 VtoF_ViewDirection;
varying mediump vec3 VtoF_UntransformedViewDirection;
varying mediump vec3 VtoF_Normal;
#endif

#ifndef NO_OBJECT_COLOUR
varying mediump vec4 VtoF_Ambient;
varying mediump vec4 VtoF_Colour;						// A combination of diffuse colour and vertex colour
#endif

#ifdef TEXTURING_0
varying mediump vec2 VtoF_TextureCoord;
#endif

#if (defined TEXTURING_1_SRCA_1MINUSSRCA) || (defined TEXTURING_1_1_SRCA) || (defined TEXTURING_1_CROSSFADE)
varying mediump vec2 VtoF_SecondaryTextureCoord;
#endif 

#ifdef CUBE_MAPPING
varying mediump vec3 VtoF_CubeTextureCoord;
#endif

#ifdef DIRECTIONAL_LIGHT_0
varying mediump vec3 VtoF_DirectionalLight0_Dir;
varying highp vec3 VtoF_DirectionalLight0_HalfVector;
#endif

#ifdef POINT_LIGHT_0
varying highp vec3 VtoF_PointSpotLight0_Dir;
varying highp vec3 VtoF_PointSpotLight0_HalfVector;
#endif

//================================================================================
//================================================================================

void main()
{
    vec3 D0lightDir;
    vec3 P0lightDir;

    vec4 newModelPosition = INPUT_ModelMatrix * ATTRIB_Vertex;
    vec4 newPosition = INPUT_ModelProjMatrix * ATTRIB_Vertex;
   
#ifdef LIGHTING_ENABLED
	mediump mat3 NormalView3 = mat3(INPUT_NormalMatrix[0].xyz , INPUT_NormalMatrix[1].xyz, INPUT_NormalMatrix[2].xyz);
	vec3 newNormal = vec3(normalize(NormalView3 * ATTRIB_Normals));
//	vec3 newNormal = vec3(normalize(NormalView3 * vec3(0.5,0.5,0.5)));
#endif

#ifdef NORMAL_MAPPING
    vec3 n = newNormal;
    vec3 t = normalize(ATTRIB_Tangents.xyz);
    vec3 b = cross(n, t) * ATTRIB_Tangents.w; // * gl_MultiTexCoord1.w;
    mat3 tbnMatrix = mat3(t.x, b.x, n.x,
                          t.y, b.y, n.y,
                          t.z, b.z, n.z);

#ifdef LIGHTING_ENABLED
	VtoF_UntransformedViewDirection = vec3(newModelPosition.xyz);                           
	VtoF_ViewDirection = tbnMatrix * vec3(newModelPosition.xyz);
#endif
	
#ifdef DIRECTIONAL_LIGHT_0
	// Directional Light - transform the light direction by the normal matrix
	// to align it with the surface normal...(tangent space)... 
	// Then then normal map texture provides deviations from that normal.. 
	//===================
	D0lightDir = normalize(vec3(INPUT_DirectionalLight0Direction));
   	VtoF_DirectionalLight0_Dir = tbnMatrix * D0lightDir;
   	VtoF_DirectionalLight0_HalfVector = normalize(tbnMatrix * INPUT_DirectionalLight0HalfVector);
#endif

#ifdef POINT_LIGHT_0
	// Point/Spot Light
	//===================
	P0lightDir = -normalize(INPUT_PointSpotLight0Position - vec3(newModelPosition));
   	VtoF_PointSpotLight0_Dir = tbnMatrix * P0lightDir;
   	VtoF_PointSpotLight0_HalfVector = normalize(tbnMatrix * INPUT_DirectionalLight0HalfVector);
#endif

#else // No Normal Mapping

#ifdef LIGHTING_ENABLED
	VtoF_ViewDirection = vec3(newModelPosition.xyz); //vec3(0,0,1);
#endif

#ifdef DIRECTIONAL_LIGHT_0
	// Directional Light
	//===================
	D0lightDir = normalize(vec3(INPUT_DirectionalLight0Direction));
   	VtoF_DirectionalLight0_Dir = D0lightDir;
   	VtoF_DirectionalLight0_HalfVector = normalize(INPUT_DirectionalLight0HalfVector);
#endif

#ifdef POINT_LIGHT_0
	// Point/Spot Light
	//===================
	P0lightDir = -normalize(INPUT_PointSpotLight0Position - vec3(newModelPosition));
   	VtoF_PointSpotLight0_Dir = P0lightDir;
   	VtoF_PointSpotLight0_HalfVector = normalize(INPUT_DirectionalLight0HalfVector);
#endif

#endif
    
#ifdef CUBE_MAPPING
	P0lightDir = -normalize(vec3(0,0,-1) - vec3(newModelPosition));
	VtoF_CubeTextureCoord.xyz = reflect( P0lightDir, newNormal ); 
#endif
    
#ifdef LIGHTING_ENABLED
	// Normals are of no interest if we're not doing lighting....
    VtoF_Normal = newNormal;
#endif

#ifndef NO_OBJECT_COLOUR
	VtoF_Ambient = INPUT_AmbientColour; 
    VtoF_Colour = INPUT_DiffuseColour * ATTRIB_Colour;
#endif

#ifdef TEXTURING_0
    VtoF_TextureCoord = ATTRIB_TextureUV;
#endif

#if (defined TEXTURING_1_SRCA_1MINUSSRCA) || (defined TEXTURING_1_1_SRCA) || (defined TEXTURING_1_CROSSFADE)
	 VtoF_SecondaryTextureCoord = ATTRIB_SecondaryTextureUV;
#endif 
    
    gl_Position = newPosition;
}
