uniform vec3 targetColor;
uniform float hueThreshold; // = 0.25
uniform float shadowThreshold; // = 0.25

const float borderSoftness = 0.05; // it may be exposed as a uniform

/// Convert RGB color to HSV.
///
/// Parameters:
/// - `rgb`: the color to convert
vec3 hsv(vec3 rgb) {
    float r = rgb.r;
    float g = rgb.g;
    float b = rgb.b;
    float cmax = max(r, max(g, b));
    float cmin = min(r, min(g, b));
    float diff = cmax - cmin;
    float h = 0.;
    float s = 0.;
    float v = 0.;
    // Compute H
    if (cmax == cmin) {
        h = 0.;
    } else if (cmax == r) {
        h = mod(60. * (g - b) / diff + 360., 360.);
    } else if (cmax == g) {
        h = mod(60. * (b - r) / diff + 120., 360.);
    } else if (cmax == b) {
        h = mod(60. * (r - g) / diff + 240., 360.);
    }
    // Compute S
    if (cmax == 0.) {
        s = 0.;
    } else {
        s = diff / cmax * 100.;
    }
    // Compute V
    v = cmax * 100.;
    return vec3(h, s, v);
}

// Possibly a faster function due to less conditionals, TODO: benchmark
//vec3 rgb2hsv(in vec3 c){
//    vec4 K = vec4(0., -1. / 3., 2. / 3., -1.);
//    vec4 p = mix(
//        vec4(c.bg, K.wz),
//        vec4(c.gb, K.xy),
//        step(c.b, c.g)
//    );
//    vec4 q = mix(
//        vec4(p.xyw, c.r),
//        vec4(c.r, p.yzx),
//        step(p.x, c.r)
//    );
//    float d = q.x - min(q.w, q.y);
//    float e = 1.0e-10;
//    return vec3(
//        abs(q.z + (q.w - q.y) / (6.0 * d + e)) * 360.,
//        d / (q.x + e) * 100.,
//        q.x * 100.
//    );
//}

// Computes the channel-wise distance between two HSV colors, considering a periodicity of 360 for H.
vec3 hsvDistance(vec3 color1, vec3 color2) {
    vec3 dist = abs(color1 - color2);
    dist.x = min(dist.x, 360. - dist.x);
    return dist;
}

// Remove `image` pixels whose color is "similar" to `key`.
//
// Parameters:
// - `image`: the color to which the chroma key is applied
// - `key`: the color we want to remove from `image`
// - `hThreshold`: the threshold for the hue of the color, in the range `0..180`
// - `sThreshold`: the threshold for the saturation of the color, in the range `0..100`
// - `vThreshold`: the threshold for the value (brightness) of the color, in the range `0..100`
vec4 chromaKey(vec4 image, vec3 key, float hThreshold, float sThreshold, float vThreshold) {
    vec3 threshold = vec3(hThreshold, sThreshold, vThreshold);
    vec3 imageHSV = hsv(image.rgb);
    vec3 keyHSV = hsv(key.rgb);

    // compute the distance between the current pixel color and the target color (key)
    vec3 dist = hsvDistance(imageHSV, keyHSV);

    // soften the border smoothing between two thresholds
    vec3 lowThres = threshold;
    vec3 highThres = threshold * (1. + borderSoftness) + 0.01; // add 0.01 to ensure it's higher
    // the normalized distance is a value between 0 and 1
    vec3 normDist = smoothstep(lowThres, highThres, dist);

    // Close points will have 0 for all dimensions of normDist, far point will have 1 at least in
    // one of those dimensions. Summ all demensions and cap the result to 1.0.
    image.a *= min(1.0, normDist.x + normDist.y + normDist.z);

    return image;
}

vec4 processColor(vec4 color) {
    return chromaKey(color, targetColor, hueThreshold, shadowThreshold, shadowThreshold);
}
