--- @class FeatureCenter
---

FeatureCenter = {}

pb = require "pb" 
globalPb = pb 

--- 通过一个Job条件, 查询DB, 获取结果
---
--- 示例 https://docs.corp.kuaishou.com/d/home/fcAD0zmnh_tcppJ5vEHjJXClh#section=h.pe3usan8ab7l
--- 
--- @class Job
--- - {'caculate'=\<Calculate\>, 'filter'=\<Filter\>, 'order_by'=\<OrderBy\>, 'limit'=\<int32\>}
---
--- @class Calculate
--- - {'op' = \<enum\>, 'keys' = List\<String\>, 'args' = List\<String\> [optional], 'group_by' = List\<String\>  [optional]}
--- - op 支持 LIST, SUM, COUNT, SUM_POWER, AVG, MAX, MIN, PERCENT
--- - args 是给特定op用的参数, 比如PERCENT 50/90
---
--- @class Filter
--- - oneof { SingleFilter, CombinedFilter }; 
--- - Init with function 'FeatureCenter:singleFilter()' or 'FeatureCenter:combinedAndFilters()'
---
--- @class OrderBy
--- - {'direction' = \<enum\>, 'keys' = List\<String\>}
--- - direction 支持 ASC, DESC
---
--- @eve_version 1.0
--- @param job Job 查询条件
--- @return table 示例 {'code'='0-成功 1-失败', 'msg'='失败原因', 'result'='查询结果'}
function FeatureCenter:calculate(job) 
    local tracker = FeatureCenterTracker:startTrack(job)
    local bytes = pb.encode("com.kwai.sdk.eve.proto.feature.Job", job) 
    local ret = featureCenterCalculate(bytes) 
    tracker:endTrack()
    return ret
end 

function FeatureCenter:exactCalculate(taskId, job) 
    local tracker = FeatureCenterTracker:startTrack(job)
    local bytes = pb.encode("com.kwai.sdk.eve.proto.feature.Job", job) 
    local ret = featureCenterExactCalculate(taskId, bytes) 
    tracker:endTrack()
    return ret
end 

function FeatureCenter:multiCalculate(jobs) 
    local tracker = FeatureCenterTracker:startTrack(job)
    local bytesList = {} 
    for i = 1, #jobs do 
        table.insert(bytesList, pb.encode("com.kwai.sdk.eve.proto.feature.Job", jobs[i])) 
    end 
    local ret = featureCenterMultiCalculate(bytesList) 
    tracker:endTrack()
    return ret
end 

function FeatureCenter:exactMultiCalculate(taskId, jobs) 
    local tracker = FeatureCenterTracker:startTrack(job)
    local bytesList = {} 
    for i = 1, #jobs do 
        table.insert(bytesList, pb.encode("com.kwai.sdk.eve.proto.feature.Job", jobs[i])) 
    end 
    local ret = featureCenterExactMultiCalculate(taskId, bytesList) 
    tracker:endTrack()
    return ret
end

--- 构造一个过滤器
--- 
--- @class Value
--- - Init with function 'FeatureCenter:primitive2Value()'
---
--- @class FilterOp
--- - 包含 'EQ', 'IN', 'BETWEEN', 'GT', 'LT', 'GE', 'LE', 'NE'
--- - 详情查看 https://git.corp.kuaishou.com/edge-reco/eve-runtime/-/blob/master/proto/job.proto
---
--- @eve_version 1.0
--- @param data table 示例 {'key'=<String>,'value'=<Value>,'op'=<FilterOp>}
--- @return Filter
function FeatureCenter:singleFilter(data) 
    return { 
        single_filter = data 
    } 
end 

--- 将过滤器按照 'AND' 的关系组合成一个更大的过滤器
---
--- @eve_version 1.0
--- @param filters List 多个singleFilter组合
--- - 示例 
--- ```lua
--- FeatureCenter:combinedAndFilters(FeatureCenter:singleFilter(data1), FeatureCenter:singleFilter(data2))
--- ```
---@return Filter
function FeatureCenter:combinedAndFilters(...) 
    return { 
        combined_filter = { 
            combine = "AND", 
            filters = { ... } 
        } 
    } 
end 

--- 将过滤器按照 'OR' 的关系组合成一个更大的过滤器
---
--- @eve_version 1.0
--- @param filters List 多个singleFilter组合
--- - 示例 
--- ```lua
--- FeatureCenter:combinedAndFilters(FeatureCenter:singleFilter(data1), FeatureCenter:singleFilter(data2))
--- ```
---@return Filter
function FeatureCenter:combinedOrFilters(...) 
    return { 
        combined_filter = { 
            combine = "OR", 
            filters = { ... } 
        } 
    } 
end 

--- 将一个数值转换为底层查询模块能理解的描述符
---
--- @generic T
--- @eve_version 1.0
--- @param value T
--- - T type = 'string'. return { type = 'STRING', string_value = T }
--- - T type = 'number'. return { type = 'DOUBLE', double_value = T }
--- - T is table && first_type = 'string'. return { type = 'STRING_LIST', string_list_value = T }
--- - T is table && first_type = 'number'. return { type = 'DOUBLE_LIST', double_list_value = T }
--- @return table
function FeatureCenter:primitive2Value(data) 
    local data_type = type(data) 
    if data_type == 'string' then 
        return { type = 'STRING', string_value = data } 
    elseif data_type == 'number' then 
        return { type = 'DOUBLE', double_value = data } 
    end 
    if data_type ~= 'table' then 
        error('type not support' .. type(data)) 
    end 
    local first = data[1] 
    if first == nil then 
        error('empty value') 
    end 
    local first_type = type(first) 
    if first_type == 'string' then 
        return { type = 'STRING_LIST', string_list_value = data } 
    elseif first_type == 'number' then 
        return { type = 'DOUBLE_LIST', double_list_value = data } 
    end 
end

--- 获取全局缓存特征
---
--- @param key String 业务自定义字符串
--- @return table 示例 {'code'='0-成功 1-失败', 'msg'='失败原因', 'result'='查询结果'}
function FeatureCenter:get(key) 
    return featureCenterGetGlobalFeature(key) 
end 

--- decode protobuf 
---
--- @param dataMap table 示例 {'protoType'=pb full name, 'protoData'=pb encode data}
--- @return table pb解析后对应的table
function FeatureCenter:parseEvent(dataMap)
    return pb.decode(dataMap.protoType, dataMap.protoData) 
end

--- LSH 哈希算法
---
--- @eve_version 1.0
--- @param embedding List<float>
--- @return uint64_t
function FeatureCenter:lsh(embedding)
    return featureCenterLsh(embedding) 
end

--- 获取特征(值形式)
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param category : 如果是用户画像，category为"USER_PROFILE"; 从app中传输过来的feature，category为"APP"，如果是上下滑场景特征，category为"NASA_FEATURED"
--- @param featureKeys table
--- @return 特征（值形式）
function FeatureCenter:getFeature(category, featureKey) 
    if category == 'NASA_FEATURED' then 
        return EveKV:get(category, featureKey) 
    elseif string.find(featureKey, 'nasa_feed') == 1 then 
        local realCategory = 'NASA_FEATURED' 
        local realFeatureKey = string.sub(featureKey, 11, -1) 
        return EveKV:get(realCategory, realFeatureKey)
    else
        Log_d('！！！如果有多个feature需要获取，使用FeatureCenter:getFeatures比getFeature效率要高很多 ！！！')
        local paramArr = {} 
        table.insert(paramArr, featureKey) 
        local ret = FeatureCenter:baseGetFeaturesWithCache('getFeatures', category, paramArr) 
        if ret ~= nil then 
            return ret[featureKey] 
        end
        return nil
    end 
end 

--- 获取特征(Ref形式)
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param category : 如果是用户画像，category为"USER_PROFILE"; 从app中传输过来的feature，category为"APP"，如果是上下滑场景特征，category为"NASA_FEATURED"
--- @param featureKeys table
--- @return 特征（ref形式）
function FeatureCenter:getFeatureRef(category, featureKey) 
    if category == 'NASA_FEATURED' then 
        return EveKV:getRef(category, featureKey) 
    elseif string.find(featureKey, 'nasa_feed') == 1 then 
        local realCategory = 'NASA_FEATURED' 
        local realFeatureKey = string.sub(featureKey, 11, -1) 
        return EveKV:getRef(realCategory, realFeatureKey) 
    else 
        Log_d('！！！如果有多个feature需要获取，使用FeatureCenter:getFeatures比getFeature效率要高很多 ！！！')
        local paramArr = {} 
        table.insert(paramArr, featureKey) 
        local ret = FeatureCenter:baseGetFeaturesWithCache('getFeatures', category, paramArr) 
        if ret ~= nil then 
            return createRef(ret[featureKey]) 
        end 
        return nil 
    end 
end 

--- 发布特征
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param category : category为"APP"
--- @param featureKey : 特征名
--- @param value : 特征值
--- @return 
function FeatureCenter:updateFeature(category, featureKey, value)
    FeatureProvider:notifyFeatureChange(category, featureKey, value)
end

--- 获取特征
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param category : 如果是用户画像，category为"USER_PROFILE"; 其他从app中传输过来的feature，category为"APP"
--- @param featureKeys table
--- @return table
function FeatureCenter:getFeatures(category, featureKeys) 
    if category == 'NASA_FEATURED' then 
        error('NASA_FEATURED can only get feedInfo by getFeatureRef() or getFeature()') 
    end 
    return FeatureCenter:baseGetFeaturesWithCache('getFeatures', category, featureKeys) 
end 

--- 获取通用特征
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param featureKeys table
--- @return table
function FeatureCenter:getCommonFeatures(featureKeys)
    return getDataProvider('getCommonFeatures', featureKeys)
end

--- 获取用户特征
---
--- - 各个特征明细字段文档 https://docs.corp.kuaishou.com/d/home/fcACizAjHCG5UwSlmzdfmUX_m#section=h.qye8u0l8khn6
--- @eve_version 1.0
--- @param featureKeys table
--- @return table
function FeatureCenter:getUserFeatures(featureKeys)
    return getDataProvider('getUserFeatures', featureKeys)
end

--- 内部方法
function FeatureCenter:baseGetFeaturesWithCache(type, category, featureKeys) 
    local cachedValue = innerFeatureCacheGet(category, featureKeys) 
    local missedKey = {} 
    for index, key in pairs(featureKeys) do 
        if cachedValue[key] == nil then 
            table.insert(missedKey, key) 
        end 
    end 
    if #missedKey == 0 then 
        return cachedValue 
    end 
    local data = {category, missedKey} 
    local missedValue = getDataProvider(type, data) 
    local allValue = {} 
    if missedValue ~= nil then 
        for k, v in pairs(missedValue) do 
            allValue[k] = v 
        end 
    end
    if cachedValue ~= nil then 
        for k, v in pairs(cachedValue) do 
            allValue[k] = v 
        end 
    end 
    return allValue 
end

return FeatureCenter