/*
 * This is the file used for brand new databases and should contain
 * the most up-to-date schema.
 */

CREATE TABLE identifiers (
    local_id TEXT NOT NULL PRIMARY KEY,
    server_id TEXT NOT NULL UNIQUE
);

CREATE TABLE ari_registry (
    local_id TEXT NOT NULL PRIMARY KEY,
    ari TEXT NOT NULL
);

CREATE TABLE change (
    /* The ID of this row */
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* Milliseconds since epoch */
    date_created INTEGER NOT NULL,

    /* Enum for the type of change */
    change_type TEXT NOT NULL,

    /* The ID of the changed model */
    model_id TEXT NOT NULL,

    /* The type of the model (team, board, etc.) */
    model_type TEXT NOT NULL,

    /* The state of the change (pending, uploading, failed, etc.) */
    state TEXT NOT NULL DEFAULT 'PENDING',

    /* The group that this change belongs to (normal, attachment); essentially different queues */
    priority INTEGER NOT NULL DEFAULT 0,

    /* UUID we send to the server for idempotency. Generated at request time, then stable for subsequent requests. */
    request_id TEXT,

    /* The number of times we have tried to upload this change. */
    attempts INTEGER NOT NULL DEFAULT 0,

    /* If this change failed, the reason for its failure (for debugging purposes only) */
    error TEXT
);

CREATE TABLE delta (
    /* The ID of this row */
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* Reference to the sync row this delta is part of */
    change_id INTEGER NOT NULL,

    /* The field on the model that has changed */
    model_field TEXT NOT NULL,

    /* The value that the field has been changed to */
    new_value TEXT,

    /* The original value of the field */
    original_value TEXT,

    /* Ensure each delta refers back to some change */
    FOREIGN KEY(change_id) REFERENCES change(_id) ON DELETE CASCADE
);

CREATE TABLE download (
    /* The ID of this row */
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* The ID of the  sync_unit e.g. boardId, cardId */
    sync_unit_id TEXT NULL,

    /* The type of the model (card, board, etc.) */
    sync_unit TEXT NOT NULL,

    /* Unique on sync_unit_id and sync_unit */
    UNIQUE (sync_unit_id, sync_unit) ON CONFLICT IGNORE
);

CREATE TABLE download_priority (
    /* The ID of this row */
    _id INTEGER  NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* The ID of the associated entry in the Download table */
    download_id INTEGER NOT NULL,

    /* Milliseconds since epoch */
    date_created INTEGER NOT NULL,

    /* This priority was created as a member of this group*/
    group_name TEXT NOT NULL,

    /* The priority for this entry */
    download_priority REAL NOT NULL,

    /* Was this request triggered by human interaction */
    user_initiated INTEGER NOT NULL DEFAULT 0,

    /* Ensure each priority references some download */
    FOREIGN KEY(download_id) REFERENCES download(_id) ON DELETE CASCADE
);

CREATE TABLE sync_unit_state (
    /* The ID of this row */
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* Which queue are we talking about here? (Upload, Download, other?) */
    sync_unit_queue TEXT NOT NULL,

    /*  What type of SyncUnit are we talking about here (Board, Card, StarredBoards, etc...) */
    sync_unit TEXT NOT NULL,

    /* The syncunit id in question - e.g. a boardId, a cardId, or possible Null */
    sync_unit_id TEXT NULL,

    /* Fetch the timestamp from DownloadData / UploadData tables.. */
    last_queued_time INTEGER NOT NULL,

    /* When the item was removed from the queue. */
    last_dequeue_time INTEGER NOT NULL,

    /* Update this when a sync begins */
    last_start_time INTEGER NOT NULL,

    /* Update this on success */
    last_success_time INTEGER NOT NULL,

    /* Like last_success_time, but error */
    last_error_time INTEGER NOT NULL,

    /* If this sync_unit belongs to a board put that board id here */
    parent_board_id TEXT NULL,

    /* If this sync_unit belongs to a card put that cards id here */
    parent_card_id TEXT NULL,

    /* Unique on sync_unit_queue, sync_unit, and sync_unit_id */
    UNIQUE ( sync_unit_queue, sync_unit, sync_unit_id ) ON CONFLICT IGNORE
);

CREATE TABLE trello_link (

    /* The ID of this row */
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,

    /* The trello link - should be a valid trello url */
    trello_link TEXT NOT NULL,

    /* The model that this trello_link points to */
    model_type TEXT NOT NULL,

    /* the local id of the model that this trello_link points to if resolved */
    local_id TEXT NULL,

    /* the name of the model that this trello_link points to if resolved */
    name TEXT NULL,

    /* error code if we were unable to resolve this link */
    error_code INTEGER NULL,

    /* error message if we were unable to resolve this link */
    error_message TEXT NULL,

    /* Unique on trello_link */
    UNIQUE (trello_link) ON CONFLICT REPLACE,

    /* Ensure each local_id refers back to some server_id */
    FOREIGN KEY(local_id) REFERENCES identifiers(local_id) ON DELETE CASCADE
);

CREATE TABLE change_vital_stats (
    _id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
    change_id INTEGER NOT NULL,
    trace_id TEXT NOT NULL,
    capability TEXT NOT NULL,
    event_source TEXT
);

CREATE INDEX change_vital_stats_change_id ON change_vital_stats(change_id);
